/* SPDX-License-Identifier: GPL-2.0 */
#ifndef _ASM_X86_EFI_H
#define _ASM_X86_EFI_H

#include <asm/fpu/api.h>
#include <asm/processor-flags.h>
#include <asm/tlb.h>
#include <asm/nospec-branch.h>
#include <asm/mmu_context.h>
#include <asm/ibt.h>
#include <linux/build_bug.h>
#include <linux/kernel.h>
#include <linux/pgtable.h>

extern unsigned long efi_fw_vendor, efi_config_table;
extern unsigned long efi_mixed_mode_stack_pa;



#define EFI32_LOADER_SIGNATURE	"EL32"
#define EFI64_LOADER_SIGNATURE	"EL64"

#define ARCH_EFI_IRQ_FLAGS_MASK	X86_EFLAGS_IF

#define EFI_UNACCEPTED_UNIT_SIZE PMD_SIZE



#define __efi_nargs(...) __efi_nargs_(__VA_ARGS__)
#define __efi_nargs_(...) __efi_nargs__(0, ##__VA_ARGS__,	\
	__efi_arg_sentinel(9), __efi_arg_sentinel(8),		\
	__efi_arg_sentinel(7), __efi_arg_sentinel(6),		\
	__efi_arg_sentinel(5), __efi_arg_sentinel(4),		\
	__efi_arg_sentinel(3), __efi_arg_sentinel(2),		\
	__efi_arg_sentinel(1), __efi_arg_sentinel(0))
#define __efi_nargs__(_0, _1, _2, _3, _4, _5, _6, _7, _8, _9, n, ...)	\
	__take_second_arg(n,					\
		({ BUILD_BUG_ON_MSG(1, "__efi_nargs limit exceeded"); 10; }))
#define __efi_arg_sentinel(n) , n



#define __efi_nargs_check(f, n, ...)					\
	__efi_nargs_check_(f, __efi_nargs(__VA_ARGS__), n)
#define __efi_nargs_check_(f, p, n) __efi_nargs_check__(f, p, n)
#define __efi_nargs_check__(f, p, n) ({					\
	BUILD_BUG_ON_MSG(						\
		(p) > (n),						\
		#f " called with too many arguments (" #p ">" #n ")");	\
})

static inline void efi_fpu_begin(void)
{
	
	kernel_fpu_begin_mask(KFPU_387 | KFPU_MXCSR);
}

static inline void efi_fpu_end(void)
{
	kernel_fpu_end();
}

#ifdef CONFIG_X86_32
#define EFI_X86_KERNEL_ALLOC_LIMIT		(SZ_512M - 1)
#else 
#define EFI_X86_KERNEL_ALLOC_LIMIT		EFI_ALLOC_LIMIT

extern asmlinkage u64 __efi_call(void *fp, ...);

extern bool efi_disable_ibt_for_runtime;

#define efi_call(...) ({						\
	__efi_nargs_check(efi_call, 7, __VA_ARGS__);			\
	__efi_call(__VA_ARGS__);					\
})

#undef arch_efi_call_virt
#define arch_efi_call_virt(p, f, args...) ({				\
	u64 ret, ibt = ibt_save(efi_disable_ibt_for_runtime);		\
	ret = efi_call((void *)p->f, args);				\
	ibt_restore(ibt);						\
	ret;								\
})

#ifdef CONFIG_KASAN

#undef memcpy
#undef memset
#undef memmove
#endif

#endif 

extern int __init efi_memblock_x86_reserve_range(void);
extern void __init efi_print_memmap(void);
extern void __init efi_map_region(efi_memory_desc_t *md);
extern void __init efi_map_region_fixed(efi_memory_desc_t *md);
extern void efi_sync_low_kernel_mappings(void);
extern int __init efi_alloc_page_tables(void);
extern int __init efi_setup_page_tables(unsigned long pa_memmap, unsigned num_pages);
extern void __init efi_runtime_update_mappings(void);
extern void __init efi_dump_pagetable(void);
extern void __init efi_apply_memmap_quirks(void);
extern int __init efi_reuse_config(u64 tables, int nr_tables);
extern void efi_delete_dummy_variable(void);
extern void efi_crash_gracefully_on_page_fault(unsigned long phys_addr);
extern void efi_free_boot_services(void);

void arch_efi_call_virt_setup(void);
void arch_efi_call_virt_teardown(void);

extern u64 efi_setup;

#ifdef CONFIG_EFI
extern u64 __efi64_thunk(u32, ...);

#define efi64_thunk(...) ({						\
	u64 __pad[3]; 	\
	__efi_nargs_check(efi64_thunk, 9, __VA_ARGS__);			\
	__efi64_thunk(__VA_ARGS__, __pad);				\
})

static inline bool efi_is_mixed(void)
{
	if (!IS_ENABLED(CONFIG_EFI_MIXED))
		return false;
	return IS_ENABLED(CONFIG_X86_64) && !efi_enabled(EFI_64BIT);
}

static inline bool efi_runtime_supported(void)
{
	if (IS_ENABLED(CONFIG_X86_64) == efi_enabled(EFI_64BIT))
		return true;

	return IS_ENABLED(CONFIG_EFI_MIXED);
}

extern void parse_efi_setup(u64 phys_addr, u32 data_len);

extern void efi_thunk_runtime_setup(void);
efi_status_t efi_set_virtual_address_map(unsigned long memory_map_size,
					 unsigned long descriptor_size,
					 u32 descriptor_version,
					 efi_memory_desc_t *virtual_map,
					 unsigned long systab_phys);



#ifdef CONFIG_EFI_MIXED

#define EFI_ALLOC_LIMIT		(efi_is_64bit() ? ULONG_MAX : U32_MAX)

#define ARCH_HAS_EFISTUB_WRAPPERS

static inline bool efi_is_64bit(void)
{
	extern const bool efi_is64;

	return efi_is64;
}

static inline bool efi_is_native(void)
{
	return efi_is_64bit();
}

#define efi_table_attr(inst, attr)					\
	(efi_is_native() ? (inst)->attr					\
			 : efi_mixed_table_attr((inst), attr))

#define efi_mixed_table_attr(inst, attr)				\
	(__typeof__(inst->attr))					\
		_Generic(inst->mixed_mode.attr,				\
		u32:		(unsigned long)(inst->mixed_mode.attr),	\
		default:	(inst->mixed_mode.attr))



static inline void *efi64_zero_upper(void *p)
{
	if (p)
		((u32 *)p)[1] = 0;
	return p;
}

static inline u32 efi64_convert_status(efi_status_t status)
{
	return (u32)(status | (u64)status >> 32);
}

#define __efi64_split(val)		(val) & U32_MAX, (u64)(val) >> 32

#define __efi64_argmap_free_pages(addr, size)				\
	((addr), 0, (size))

#define __efi64_argmap_get_memory_map(mm_size, mm, key, size, ver)	\
	((mm_size), (mm), efi64_zero_upper(key), efi64_zero_upper(size), (ver))

#define __efi64_argmap_allocate_pool(type, size, buffer)		\
	((type), (size), efi64_zero_upper(buffer))

#define __efi64_argmap_create_event(type, tpl, f, c, event)		\
	((type), (tpl), (f), (c), efi64_zero_upper(event))

#define __efi64_argmap_set_timer(event, type, time)			\
	((event), (type), lower_32_bits(time), upper_32_bits(time))

#define __efi64_argmap_wait_for_event(num, event, index)		\
	((num), (event), efi64_zero_upper(index))

#define __efi64_argmap_handle_protocol(handle, protocol, interface)	\
	((handle), (protocol), efi64_zero_upper(interface))

#define __efi64_argmap_locate_protocol(protocol, reg, interface)	\
	((protocol), (reg), efi64_zero_upper(interface))

#define __efi64_argmap_locate_device_path(protocol, path, handle)	\
	((protocol), (path), efi64_zero_upper(handle))

#define __efi64_argmap_exit(handle, status, size, data)			\
	((handle), efi64_convert_status(status), (size), (data))


#define __efi64_argmap_get_location(protocol, seg, bus, dev, func)	\
	((protocol), efi64_zero_upper(seg), efi64_zero_upper(bus),	\
	 efi64_zero_upper(dev), efi64_zero_upper(func))


#define __efi64_argmap_load_file(protocol, path, policy, bufsize, buf)	\
	((protocol), (path), (policy), efi64_zero_upper(bufsize), (buf))


#define __efi64_argmap_query_mode(gop, mode, size, info)		\
	((gop), (mode), efi64_zero_upper(size), efi64_zero_upper(info))


#define __efi64_argmap_hash_log_extend_event(prot, fl, addr, size, ev)	\
	((prot), (fl), 0ULL, (u64)(addr), 0ULL, (u64)(size), 0ULL, ev)


#define __efi64_argmap_get_memory_space_descriptor(phys, desc) \
	(__efi64_split(phys), (desc))

#define __efi64_argmap_set_memory_space_attributes(phys, size, flags) \
	(__efi64_split(phys), __efi64_split(size), __efi64_split(flags))


#define __efi64_argmap_open(prot, newh, fname, mode, attr) \
	((prot), efi64_zero_upper(newh), (fname), __efi64_split(mode), \
	 __efi64_split(attr))

#define __efi64_argmap_set_position(pos) (__efi64_split(pos))


#define __efi64_argmap_open_volume(prot, file) \
	((prot), efi64_zero_upper(file))


#define __efi64_argmap_get_memory_attributes(protocol, phys, size, flags) \
	((protocol), __efi64_split(phys), __efi64_split(size), (flags))

#define __efi64_argmap_set_memory_attributes(protocol, phys, size, flags) \
	((protocol), __efi64_split(phys), __efi64_split(size), __efi64_split(flags))

#define __efi64_argmap_clear_memory_attributes(protocol, phys, size, flags) \
	((protocol), __efi64_split(phys), __efi64_split(size), __efi64_split(flags))


#define __efi64_argmap_get_next(protocol, smbioshandle, type, record, phandle) \
	((protocol), (smbioshandle), (type), efi64_zero_upper(record), \
	 efi64_zero_upper(phandle))


#define __efi64_thunk_map(inst, func, ...)				\
	efi64_thunk(inst->mixed_mode.func,				\
		__efi64_argmap(__efi64_argmap_ ## func(__VA_ARGS__),	\
			       (__VA_ARGS__)))

#define __efi64_argmap(mapped, args)					\
	__PASTE(__efi64_argmap__, __efi_nargs(__efi_eat mapped))(mapped, args)
#define __efi64_argmap__0(mapped, args) __efi_eval mapped
#define __efi64_argmap__1(mapped, args) __efi_eval args

#define __efi_eat(...)
#define __efi_eval(...) __VA_ARGS__

static inline efi_status_t __efi64_widen_efi_status(u64 status)
{
	
	return ror64(rol32(status, 1), 1);
}



#define efi_fn_call(inst, func, ...)					\
	(efi_is_native() ? (inst)->func(__VA_ARGS__)			\
			 : efi_mixed_call((inst), func, ##__VA_ARGS__))

#define efi_mixed_call(inst, func, ...)					\
	_Generic(inst->func(__VA_ARGS__),				\
	efi_status_t:							\
		__efi64_widen_efi_status(				\
			__efi64_thunk_map(inst, func, ##__VA_ARGS__)),	\
	u64: ({ BUILD_BUG(); ULONG_MAX; }),				\
	default:							\
		(__typeof__(inst->func(__VA_ARGS__)))			\
			__efi64_thunk_map(inst, func, ##__VA_ARGS__))

#else 

static inline bool efi_is_64bit(void)
{
	return IS_ENABLED(CONFIG_X86_64);
}

#endif 

extern bool efi_reboot_required(void);
extern bool efi_is_table_address(unsigned long phys_addr);

extern void efi_reserve_boot_services(void);
#else
static inline void parse_efi_setup(u64 phys_addr, u32 data_len) {}
static inline bool efi_reboot_required(void)
{
	return false;
}
static inline  bool efi_is_table_address(unsigned long phys_addr)
{
	return false;
}
static inline void efi_reserve_boot_services(void)
{
}
#endif 

extern int __init efi_memmap_alloc(unsigned int num_entries,
				   struct efi_memory_map_data *data);

extern int __init efi_memmap_install(struct efi_memory_map_data *data);
extern int __init efi_memmap_split_count(efi_memory_desc_t *md,
					 struct range *range);
extern void __init efi_memmap_insert(struct efi_memory_map *old_memmap,
				     void *buf, struct efi_mem_range *mem);

extern enum efi_secureboot_mode __x86_ima_efi_boot_mode(void);

#define arch_ima_efi_boot_mode	__x86_ima_efi_boot_mode()

#ifdef CONFIG_EFI_RUNTIME_MAP
int efi_get_runtime_map_size(void);
int efi_get_runtime_map_desc_size(void);
int efi_runtime_map_copy(void *buf, size_t bufsz);
#else
static inline int efi_get_runtime_map_size(void)
{
	return 0;
}

static inline int efi_get_runtime_map_desc_size(void)
{
	return 0;
}

static inline int efi_runtime_map_copy(void *buf, size_t bufsz)
{
	return 0;
}

#endif

#endif 
