/* SPDX-License-Identifier: GPL-2.0-only */


#ifndef LINUX_IEEE802154_H
#define LINUX_IEEE802154_H

#include <linux/types.h>
#include <linux/random.h>

#define IEEE802154_MTU			127
#define IEEE802154_ACK_PSDU_LEN		5
#define IEEE802154_MIN_PSDU_LEN		9
#define IEEE802154_FCS_LEN		2
#define IEEE802154_MAX_AUTH_TAG_LEN	16
#define IEEE802154_FC_LEN		2
#define IEEE802154_SEQ_LEN		1


#define IEEE802154_MAX_HEADER_LEN	(2 + 1 + 20 + 14)
#define IEEE802154_MIN_HEADER_LEN	(IEEE802154_ACK_PSDU_LEN - \
					 IEEE802154_FCS_LEN)

#define IEEE802154_PAN_ID_BROADCAST	0xffff
#define IEEE802154_ADDR_SHORT_BROADCAST	0xffff
#define IEEE802154_ADDR_SHORT_UNSPEC	0xfffe

#define IEEE802154_EXTENDED_ADDR_LEN	8
#define IEEE802154_SHORT_ADDR_LEN	2
#define IEEE802154_PAN_ID_LEN		2


#define IEEE802154_MAX_SCAN_DURATION	14
#define IEEE802154_ACTIVE_SCAN_DURATION	15

#define IEEE802154_SUPERFRAME_PERIOD	16

#define IEEE802154_SLOT_PERIOD		60
#define IEEE802154_LIFS_PERIOD		40
#define IEEE802154_SIFS_PERIOD		12
#define IEEE802154_MAX_SIFS_FRAME_SIZE	18

#define IEEE802154_MAX_CHANNEL		26
#define IEEE802154_MAX_PAGE		31

#define IEEE802154_FC_TYPE_BEACON	0x0	
#define	IEEE802154_FC_TYPE_DATA		0x1	
#define IEEE802154_FC_TYPE_ACK		0x2	
#define IEEE802154_FC_TYPE_MAC_CMD	0x3	

#define IEEE802154_FC_TYPE_SHIFT		0
#define IEEE802154_FC_TYPE_MASK		((1 << 3) - 1)
#define IEEE802154_FC_TYPE(x)		((x & IEEE802154_FC_TYPE_MASK) >> IEEE802154_FC_TYPE_SHIFT)
#define IEEE802154_FC_SET_TYPE(v, x)	do {	\
	v = (((v) & ~IEEE802154_FC_TYPE_MASK) | \
	    (((x) << IEEE802154_FC_TYPE_SHIFT) & IEEE802154_FC_TYPE_MASK)); \
	} while (0)

#define IEEE802154_FC_SECEN_SHIFT	3
#define IEEE802154_FC_SECEN		(1 << IEEE802154_FC_SECEN_SHIFT)
#define IEEE802154_FC_FRPEND_SHIFT	4
#define IEEE802154_FC_FRPEND		(1 << IEEE802154_FC_FRPEND_SHIFT)
#define IEEE802154_FC_ACK_REQ_SHIFT	5
#define IEEE802154_FC_ACK_REQ		(1 << IEEE802154_FC_ACK_REQ_SHIFT)
#define IEEE802154_FC_INTRA_PAN_SHIFT	6
#define IEEE802154_FC_INTRA_PAN		(1 << IEEE802154_FC_INTRA_PAN_SHIFT)

#define IEEE802154_FC_SAMODE_SHIFT	14
#define IEEE802154_FC_SAMODE_MASK	(3 << IEEE802154_FC_SAMODE_SHIFT)
#define IEEE802154_FC_DAMODE_SHIFT	10
#define IEEE802154_FC_DAMODE_MASK	(3 << IEEE802154_FC_DAMODE_SHIFT)

#define IEEE802154_FC_VERSION_SHIFT	12
#define IEEE802154_FC_VERSION_MASK	(3 << IEEE802154_FC_VERSION_SHIFT)
#define IEEE802154_FC_VERSION(x)	((x & IEEE802154_FC_VERSION_MASK) >> IEEE802154_FC_VERSION_SHIFT)

#define IEEE802154_FC_SAMODE(x)		\
	(((x) & IEEE802154_FC_SAMODE_MASK) >> IEEE802154_FC_SAMODE_SHIFT)

#define IEEE802154_FC_DAMODE(x)		\
	(((x) & IEEE802154_FC_DAMODE_MASK) >> IEEE802154_FC_DAMODE_SHIFT)

#define IEEE802154_SCF_SECLEVEL_MASK		7
#define IEEE802154_SCF_SECLEVEL_SHIFT		0
#define IEEE802154_SCF_SECLEVEL(x)		(x & IEEE802154_SCF_SECLEVEL_MASK)
#define IEEE802154_SCF_KEY_ID_MODE_SHIFT	3
#define IEEE802154_SCF_KEY_ID_MODE_MASK		(3 << IEEE802154_SCF_KEY_ID_MODE_SHIFT)
#define IEEE802154_SCF_KEY_ID_MODE(x)		\
	((x & IEEE802154_SCF_KEY_ID_MODE_MASK) >> IEEE802154_SCF_KEY_ID_MODE_SHIFT)

#define IEEE802154_SCF_KEY_IMPLICIT		0
#define IEEE802154_SCF_KEY_INDEX		1
#define IEEE802154_SCF_KEY_SHORT_INDEX		2
#define IEEE802154_SCF_KEY_HW_INDEX		3

#define IEEE802154_SCF_SECLEVEL_NONE		0
#define IEEE802154_SCF_SECLEVEL_MIC32		1
#define IEEE802154_SCF_SECLEVEL_MIC64		2
#define IEEE802154_SCF_SECLEVEL_MIC128		3
#define IEEE802154_SCF_SECLEVEL_ENC		4
#define IEEE802154_SCF_SECLEVEL_ENC_MIC32	5
#define IEEE802154_SCF_SECLEVEL_ENC_MIC64	6
#define IEEE802154_SCF_SECLEVEL_ENC_MIC128	7


#define IEEE802154_MFR_SIZE	2 


#define IEEE802154_CMD_ASSOCIATION_REQ		0x01
#define IEEE802154_CMD_ASSOCIATION_RESP		0x02
#define IEEE802154_CMD_DISASSOCIATION_NOTIFY	0x03
#define IEEE802154_CMD_DATA_REQ			0x04
#define IEEE802154_CMD_PANID_CONFLICT_NOTIFY	0x05
#define IEEE802154_CMD_ORPHAN_NOTIFY		0x06
#define IEEE802154_CMD_BEACON_REQ		0x07
#define IEEE802154_CMD_COORD_REALIGN_NOTIFY	0x08
#define IEEE802154_CMD_GTS_REQ			0x09


enum {
	
	IEEE802154_SUCCESS = 0x0,
	
	IEEE802154_MAC_ERROR = 0x1,
	
	IEEE802154_CANCELLED = 0x2,
	
	IEEE802154_READY_FOR_POLL = 0x3,
	
	IEEE802154_COUNTER_ERROR = 0xdb,
	
	IEEE802154_IMPROPER_KEY_TYPE = 0xdc,
	
	IEEE802154_IMPROPER_SECURITY_LEVEL = 0xdd,
	
	IEEE802154_UNSUPPORTED_LEGACY = 0xde,
	
	IEEE802154_UNSUPPORTED_SECURITY = 0xdf,
	
	IEEE802154_BEACON_LOST = 0xe0,
	
	IEEE802154_CHANNEL_ACCESS_FAILURE = 0xe1,
	
	IEEE802154_DENIED = 0xe2,
	
	IEEE802154_DISABLE_TRX_FAILURE = 0xe3,
	
	IEEE802154_FAILED_SECURITY_CHECK = 0xe4,
	
	IEEE802154_FRAME_TOO_LONG = 0xe5,
	
	IEEE802154_INVALID_GTS = 0xe6,
	
	IEEE802154_INVALID_HANDLE = 0xe7,
	
	IEEE802154_INVALID_PARAMETER = 0xe8,
	
	IEEE802154_NO_ACK = 0xe9,
	
	IEEE802154_NO_BEACON = 0xea,
	
	IEEE802154_NO_DATA = 0xeb,
	
	IEEE802154_NO_SHORT_ADDRESS = 0xec,
	
	IEEE802154_OUT_OF_CAP = 0xed,
	
	IEEE802154_PAN_ID_CONFLICT = 0xee,
	
	IEEE802154_REALIGNMENT = 0xef,
	
	IEEE802154_TRANSACTION_EXPIRED = 0xf0,
	
	IEEE802154_TRANSACTION_OVERFLOW = 0xf1,
	
	IEEE802154_TX_ACTIVE = 0xf2,
	
	IEEE802154_UNAVAILABLE_KEY = 0xf3,
	
	IEEE802154_UNSUPPORTED_ATTRIBUTE = 0xf4,
	
	IEEE802154_INVALID_ADDRESS = 0xf5,
	
	IEEE802154_ON_TIME_TOO_LONG = 0xf6,
	
	IEEE802154_PAST_TIME = 0xf7,
	
	IEEE802154_TRACKING_OFF = 0xf8,
	
	IEEE802154_INVALID_INDEX = 0xf9,
	
	IEEE802154_LIMIT_REACHED = 0xfa,
	
	IEEE802154_READ_ONLY = 0xfb,
	
	IEEE802154_SCAN_IN_PROGRESS = 0xfc,
	
	IEEE802154_SUPERFRAME_OVERLAP = 0xfd,
	
	IEEE802154_SYSTEM_ERROR = 0xff,
};


enum ieee802154_filtering_level {
	IEEE802154_FILTERING_NONE,
	IEEE802154_FILTERING_1_FCS,
	IEEE802154_FILTERING_2_PROMISCUOUS,
	IEEE802154_FILTERING_3_SCAN,
	IEEE802154_FILTERING_4_FRAME_FIELDS,
};


#define IEEE802154_FCTL_FTYPE		0x0003
#define IEEE802154_FCTL_ACKREQ		0x0020
#define IEEE802154_FCTL_SECEN		0x0004
#define IEEE802154_FCTL_INTRA_PAN	0x0040
#define IEEE802154_FCTL_DADDR		0x0c00
#define IEEE802154_FCTL_SADDR		0xc000

#define IEEE802154_FTYPE_DATA		0x0001

#define IEEE802154_FCTL_ADDR_NONE	0x0000
#define IEEE802154_FCTL_DADDR_SHORT	0x0800
#define IEEE802154_FCTL_DADDR_EXTENDED	0x0c00
#define IEEE802154_FCTL_SADDR_SHORT	0x8000
#define IEEE802154_FCTL_SADDR_EXTENDED	0xc000


static inline int ieee802154_is_data(__le16 fc)
{
	return (fc & cpu_to_le16(IEEE802154_FCTL_FTYPE)) ==
		cpu_to_le16(IEEE802154_FTYPE_DATA);
}


static inline bool ieee802154_is_secen(__le16 fc)
{
	return fc & cpu_to_le16(IEEE802154_FCTL_SECEN);
}


static inline bool ieee802154_is_ackreq(__le16 fc)
{
	return fc & cpu_to_le16(IEEE802154_FCTL_ACKREQ);
}


static inline bool ieee802154_is_intra_pan(__le16 fc)
{
	return fc & cpu_to_le16(IEEE802154_FCTL_INTRA_PAN);
}


static inline __le16 ieee802154_daddr_mode(__le16 fc)
{
	return fc & cpu_to_le16(IEEE802154_FCTL_DADDR);
}


static inline __le16 ieee802154_saddr_mode(__le16 fc)
{
	return fc & cpu_to_le16(IEEE802154_FCTL_SADDR);
}


static inline bool ieee802154_is_valid_psdu_len(u8 len)
{
	return (len == IEEE802154_ACK_PSDU_LEN ||
		(len >= IEEE802154_MIN_PSDU_LEN && len <= IEEE802154_MTU));
}


static inline bool ieee802154_is_valid_extended_unicast_addr(__le64 addr)
{
	
	return ((addr != cpu_to_le64(0x0000000000000000ULL)) &&
		!(addr & cpu_to_le64(0x0100000000000000ULL)));
}


static inline bool ieee802154_is_broadcast_short_addr(__le16 addr)
{
	return (addr == cpu_to_le16(IEEE802154_ADDR_SHORT_BROADCAST));
}


static inline bool ieee802154_is_unspec_short_addr(__le16 addr)
{
	return (addr == cpu_to_le16(IEEE802154_ADDR_SHORT_UNSPEC));
}


static inline bool ieee802154_is_valid_src_short_addr(__le16 addr)
{
	return !(ieee802154_is_broadcast_short_addr(addr) ||
		 ieee802154_is_unspec_short_addr(addr));
}


static inline void ieee802154_random_extended_addr(__le64 *addr)
{
	get_random_bytes(addr, IEEE802154_EXTENDED_ADDR_LEN);

	
	((u8 *)addr)[IEEE802154_EXTENDED_ADDR_LEN - 1] &= ~0x01;
	((u8 *)addr)[IEEE802154_EXTENDED_ADDR_LEN - 1] |= 0x02;
}

#endif 
