/* SPDX-License-Identifier: GPL-2.0-or-later */
#ifndef _IIO_BACKEND_H_
#define _IIO_BACKEND_H_

#include <linux/types.h>
#include <linux/iio/iio.h>

struct iio_chan_spec;
struct fwnode_handle;
struct iio_backend;
struct device;
struct iio_dev;

enum iio_backend_data_type {
	IIO_BACKEND_TWOS_COMPLEMENT,
	IIO_BACKEND_OFFSET_BINARY,
	IIO_BACKEND_DATA_UNSIGNED,
	IIO_BACKEND_DATA_TYPE_MAX
};

enum iio_backend_data_source {
	IIO_BACKEND_INTERNAL_CONTINUOUS_WAVE,
	IIO_BACKEND_EXTERNAL,
	IIO_BACKEND_INTERNAL_RAMP_16BIT,
	IIO_BACKEND_DATA_SOURCE_MAX
};

#define iio_backend_debugfs_ptr(ptr)	PTR_IF(IS_ENABLED(CONFIG_DEBUG_FS), ptr)


#define IIO_BACKEND_EX_INFO(_name, _shared, _what) {	\
	.name = (_name),				\
	.shared = (_shared),				\
	.read =  iio_backend_ext_info_get,		\
	.write = iio_backend_ext_info_set,		\
	.private = (_what),				\
}


struct iio_backend_data_fmt {
	enum iio_backend_data_type type;
	bool sign_extend;
	bool enable;
};


enum iio_backend_test_pattern {
	IIO_BACKEND_NO_TEST_PATTERN,
	
	IIO_BACKEND_ADI_PRBS_9A = 32,
	
	IIO_BACKEND_ADI_PRBS_23A,
	IIO_BACKEND_TEST_PATTERN_MAX
};

enum iio_backend_sample_trigger {
	IIO_BACKEND_SAMPLE_TRIGGER_EDGE_FALLING,
	IIO_BACKEND_SAMPLE_TRIGGER_EDGE_RISING,
	IIO_BACKEND_SAMPLE_TRIGGER_MAX
};


struct iio_backend_ops {
	int (*enable)(struct iio_backend *back);
	void (*disable)(struct iio_backend *back);
	int (*chan_enable)(struct iio_backend *back, unsigned int chan);
	int (*chan_disable)(struct iio_backend *back, unsigned int chan);
	int (*data_format_set)(struct iio_backend *back, unsigned int chan,
			       const struct iio_backend_data_fmt *data);
	int (*data_source_set)(struct iio_backend *back, unsigned int chan,
			       enum iio_backend_data_source data);
	int (*set_sample_rate)(struct iio_backend *back, unsigned int chan,
			       u64 sample_rate_hz);
	int (*test_pattern_set)(struct iio_backend *back,
				unsigned int chan,
				enum iio_backend_test_pattern pattern);
	int (*chan_status)(struct iio_backend *back, unsigned int chan,
			   bool *error);
	int (*iodelay_set)(struct iio_backend *back, unsigned int chan,
			   unsigned int taps);
	int (*data_sample_trigger)(struct iio_backend *back,
				   enum iio_backend_sample_trigger trigger);
	struct iio_buffer *(*request_buffer)(struct iio_backend *back,
					     struct iio_dev *indio_dev);
	void (*free_buffer)(struct iio_backend *back,
			    struct iio_buffer *buffer);
	int (*extend_chan_spec)(struct iio_backend *back,
				struct iio_chan_spec *chan);
	int (*ext_info_set)(struct iio_backend *back, uintptr_t private,
			    const struct iio_chan_spec *chan,
			    const char *buf, size_t len);
	int (*ext_info_get)(struct iio_backend *back, uintptr_t private,
			    const struct iio_chan_spec *chan, char *buf);
	int (*read_raw)(struct iio_backend *back,
			struct iio_chan_spec const *chan, int *val, int *val2,
			long mask);
	int (*debugfs_print_chan_status)(struct iio_backend *back,
					 unsigned int chan, char *buf,
					 size_t len);
	int (*debugfs_reg_access)(struct iio_backend *back, unsigned int reg,
				  unsigned int writeval, unsigned int *readval);
	int (*ddr_enable)(struct iio_backend *back);
	int (*ddr_disable)(struct iio_backend *back);
	int (*data_stream_enable)(struct iio_backend *back);
	int (*data_stream_disable)(struct iio_backend *back);
	int (*data_transfer_addr)(struct iio_backend *back, u32 address);
};


struct iio_backend_info {
	const char *name;
	const struct iio_backend_ops *ops;
};

int iio_backend_chan_enable(struct iio_backend *back, unsigned int chan);
int iio_backend_chan_disable(struct iio_backend *back, unsigned int chan);
int devm_iio_backend_enable(struct device *dev, struct iio_backend *back);
int iio_backend_enable(struct iio_backend *back);
void iio_backend_disable(struct iio_backend *back);
int iio_backend_data_format_set(struct iio_backend *back, unsigned int chan,
				const struct iio_backend_data_fmt *data);
int iio_backend_data_source_set(struct iio_backend *back, unsigned int chan,
				enum iio_backend_data_source data);
int iio_backend_set_sampling_freq(struct iio_backend *back, unsigned int chan,
				  u64 sample_rate_hz);
int iio_backend_test_pattern_set(struct iio_backend *back,
				 unsigned int chan,
				 enum iio_backend_test_pattern pattern);
int iio_backend_chan_status(struct iio_backend *back, unsigned int chan,
			    bool *error);
int iio_backend_iodelay_set(struct iio_backend *back, unsigned int lane,
			    unsigned int taps);
int iio_backend_data_sample_trigger(struct iio_backend *back,
				    enum iio_backend_sample_trigger trigger);
int devm_iio_backend_request_buffer(struct device *dev,
				    struct iio_backend *back,
				    struct iio_dev *indio_dev);
int iio_backend_ddr_enable(struct iio_backend *back);
int iio_backend_ddr_disable(struct iio_backend *back);
int iio_backend_data_stream_enable(struct iio_backend *back);
int iio_backend_data_stream_disable(struct iio_backend *back);
int iio_backend_data_transfer_addr(struct iio_backend *back, u32 address);
ssize_t iio_backend_ext_info_set(struct iio_dev *indio_dev, uintptr_t private,
				 const struct iio_chan_spec *chan,
				 const char *buf, size_t len);
ssize_t iio_backend_ext_info_get(struct iio_dev *indio_dev, uintptr_t private,
				 const struct iio_chan_spec *chan, char *buf);
int iio_backend_read_raw(struct iio_backend *back,
			 struct iio_chan_spec const *chan, int *val, int *val2,
			 long mask);
int iio_backend_extend_chan_spec(struct iio_backend *back,
				 struct iio_chan_spec *chan);
void *iio_backend_get_priv(const struct iio_backend *conv);
struct iio_backend *devm_iio_backend_get(struct device *dev, const char *name);
struct iio_backend *devm_iio_backend_fwnode_get(struct device *dev,
						const char *name,
						struct fwnode_handle *fwnode);
struct iio_backend *
__devm_iio_backend_get_from_fwnode_lookup(struct device *dev,
					  struct fwnode_handle *fwnode);

int devm_iio_backend_register(struct device *dev,
			      const struct iio_backend_info *info, void *priv);

static inline int iio_backend_read_scale(struct iio_backend *back,
					 struct iio_chan_spec const *chan,
					 int *val, int *val2)
{
	return iio_backend_read_raw(back, chan, val, val2, IIO_CHAN_INFO_SCALE);
}

static inline int iio_backend_read_offset(struct iio_backend *back,
					  struct iio_chan_spec const *chan,
					  int *val, int *val2)
{
	return iio_backend_read_raw(back, chan, val, val2,
				    IIO_CHAN_INFO_OFFSET);
}

ssize_t iio_backend_debugfs_print_chan_status(struct iio_backend *back,
					      unsigned int chan, char *buf,
					      size_t len);
void iio_backend_debugfs_add(struct iio_backend *back,
			     struct iio_dev *indio_dev);
#endif
