/* SPDX-License-Identifier: GPL-2.0-or-later */


#ifndef __MTD_MTD_H__
#define __MTD_MTD_H__

#include <linux/types.h>
#include <linux/uio.h>
#include <linux/list.h>
#include <linux/notifier.h>
#include <linux/device.h>
#include <linux/of.h>
#include <linux/nvmem-provider.h>

#include <mtd/mtd-abi.h>

#include <asm/div64.h>

#define MTD_FAIL_ADDR_UNKNOWN -1LL

struct mtd_info;


struct erase_info {
	uint64_t addr;
	uint64_t len;
	uint64_t fail_addr;
};

struct mtd_erase_region_info {
	uint64_t offset;		
	uint32_t erasesize;		
	uint32_t numblocks;		
	unsigned long *lockmap;		
};

struct mtd_req_stats {
	unsigned int uncorrectable_errors;
	unsigned int corrected_bitflips;
	unsigned int max_bitflips;
};


struct mtd_oob_ops {
	unsigned int	mode;
	size_t		len;
	size_t		retlen;
	size_t		ooblen;
	size_t		oobretlen;
	uint32_t	ooboffs;
	uint8_t		*datbuf;
	uint8_t		*oobbuf;
	struct mtd_req_stats *stats;
};


struct mtd_oob_region {
	u32 offset;
	u32 length;
};


struct mtd_ooblayout_ops {
	int (*ecc)(struct mtd_info *mtd, int section,
		   struct mtd_oob_region *oobecc);
	int (*free)(struct mtd_info *mtd, int section,
		    struct mtd_oob_region *oobfree);
};


struct mtd_pairing_info {
	int pair;
	int group;
};


struct mtd_pairing_scheme {
	int ngroups;
	int (*get_info)(struct mtd_info *mtd, int wunit,
			struct mtd_pairing_info *info);
	int (*get_wunit)(struct mtd_info *mtd,
			 const struct mtd_pairing_info *info);
};

struct module;	


struct mtd_debug_info {
	struct dentry *dfs_dir;
};


struct mtd_part {
	struct list_head node;
	u64 offset;
	u64 size;
	u32 flags;
};


struct mtd_master {
	struct mutex partitions_lock;
	struct mutex chrdev_lock;
	unsigned int suspended : 1;
};

struct mtd_info {
	u_char type;
	uint32_t flags;
	uint64_t size;	 // Total size of the MTD

	
	uint32_t erasesize;
	
	uint32_t writesize;

	
	uint32_t writebufsize;

	uint32_t oobsize;   // Amount of OOB data per block (e.g. 16)
	uint32_t oobavail;  // Available OOB bytes per block

	
	unsigned int erasesize_shift;
	unsigned int writesize_shift;
	
	unsigned int erasesize_mask;
	unsigned int writesize_mask;

	
	unsigned int bitflip_threshold;

	
	const char *name;
	int index;

	
	const struct mtd_ooblayout_ops *ooblayout;

	
	const struct mtd_pairing_scheme *pairing;

	
	unsigned int ecc_step_size;

	
	unsigned int ecc_strength;

	
	int numeraseregions;
	struct mtd_erase_region_info *eraseregions;

	
	int (*_erase) (struct mtd_info *mtd, struct erase_info *instr);
	int (*_point) (struct mtd_info *mtd, loff_t from, size_t len,
		       size_t *retlen, void **virt, resource_size_t *phys);
	int (*_unpoint) (struct mtd_info *mtd, loff_t from, size_t len);
	int (*_read) (struct mtd_info *mtd, loff_t from, size_t len,
		      size_t *retlen, u_char *buf);
	int (*_write) (struct mtd_info *mtd, loff_t to, size_t len,
		       size_t *retlen, const u_char *buf);
	int (*_panic_write) (struct mtd_info *mtd, loff_t to, size_t len,
			     size_t *retlen, const u_char *buf);
	int (*_read_oob) (struct mtd_info *mtd, loff_t from,
			  struct mtd_oob_ops *ops);
	int (*_write_oob) (struct mtd_info *mtd, loff_t to,
			   struct mtd_oob_ops *ops);
	int (*_get_fact_prot_info) (struct mtd_info *mtd, size_t len,
				    size_t *retlen, struct otp_info *buf);
	int (*_read_fact_prot_reg) (struct mtd_info *mtd, loff_t from,
				    size_t len, size_t *retlen, u_char *buf);
	int (*_get_user_prot_info) (struct mtd_info *mtd, size_t len,
				    size_t *retlen, struct otp_info *buf);
	int (*_read_user_prot_reg) (struct mtd_info *mtd, loff_t from,
				    size_t len, size_t *retlen, u_char *buf);
	int (*_write_user_prot_reg) (struct mtd_info *mtd, loff_t to,
				     size_t len, size_t *retlen,
				     const u_char *buf);
	int (*_lock_user_prot_reg) (struct mtd_info *mtd, loff_t from,
				    size_t len);
	int (*_erase_user_prot_reg) (struct mtd_info *mtd, loff_t from,
				     size_t len);
	int (*_writev) (struct mtd_info *mtd, const struct kvec *vecs,
			unsigned long count, loff_t to, size_t *retlen);
	void (*_sync) (struct mtd_info *mtd);
	int (*_lock) (struct mtd_info *mtd, loff_t ofs, uint64_t len);
	int (*_unlock) (struct mtd_info *mtd, loff_t ofs, uint64_t len);
	int (*_is_locked) (struct mtd_info *mtd, loff_t ofs, uint64_t len);
	int (*_block_isreserved) (struct mtd_info *mtd, loff_t ofs);
	int (*_block_isbad) (struct mtd_info *mtd, loff_t ofs);
	int (*_block_markbad) (struct mtd_info *mtd, loff_t ofs);
	int (*_max_bad_blocks) (struct mtd_info *mtd, loff_t ofs, size_t len);
	int (*_suspend) (struct mtd_info *mtd);
	void (*_resume) (struct mtd_info *mtd);
	void (*_reboot) (struct mtd_info *mtd);
	
	int (*_get_device) (struct mtd_info *mtd);
	void (*_put_device) (struct mtd_info *mtd);

	
	bool oops_panic_write;

	struct notifier_block reboot_notifier;  

	
	struct mtd_ecc_stats ecc_stats;
	
	int subpage_sft;

	void *priv;

	struct module *owner;
	struct device dev;
	struct kref refcnt;
	struct mtd_debug_info dbg;
	struct nvmem_device *nvmem;
	struct nvmem_device *otp_user_nvmem;
	struct nvmem_device *otp_factory_nvmem;

	
	struct mtd_info *parent;

	
	struct list_head partitions;

	struct mtd_part part;
	struct mtd_master master;
};

static inline struct mtd_info *mtd_get_master(struct mtd_info *mtd)
{
	while (mtd->parent)
		mtd = mtd->parent;

	return mtd;
}

static inline u64 mtd_get_master_ofs(struct mtd_info *mtd, u64 ofs)
{
	while (mtd->parent) {
		ofs += mtd->part.offset;
		mtd = mtd->parent;
	}

	return ofs;
}

static inline bool mtd_is_partition(const struct mtd_info *mtd)
{
	return mtd->parent;
}

static inline bool mtd_has_partitions(const struct mtd_info *mtd)
{
	return !list_empty(&mtd->partitions);
}

int mtd_ooblayout_ecc(struct mtd_info *mtd, int section,
		      struct mtd_oob_region *oobecc);
int mtd_ooblayout_find_eccregion(struct mtd_info *mtd, int eccbyte,
				 int *section,
				 struct mtd_oob_region *oobregion);
int mtd_ooblayout_get_eccbytes(struct mtd_info *mtd, u8 *eccbuf,
			       const u8 *oobbuf, int start, int nbytes);
int mtd_ooblayout_set_eccbytes(struct mtd_info *mtd, const u8 *eccbuf,
			       u8 *oobbuf, int start, int nbytes);
int mtd_ooblayout_free(struct mtd_info *mtd, int section,
		       struct mtd_oob_region *oobfree);
int mtd_ooblayout_get_databytes(struct mtd_info *mtd, u8 *databuf,
				const u8 *oobbuf, int start, int nbytes);
int mtd_ooblayout_set_databytes(struct mtd_info *mtd, const u8 *databuf,
				u8 *oobbuf, int start, int nbytes);
int mtd_ooblayout_count_freebytes(struct mtd_info *mtd);
int mtd_ooblayout_count_eccbytes(struct mtd_info *mtd);

static inline void mtd_set_ooblayout(struct mtd_info *mtd,
				     const struct mtd_ooblayout_ops *ooblayout)
{
	mtd->ooblayout = ooblayout;
}

static inline void mtd_set_pairing_scheme(struct mtd_info *mtd,
				const struct mtd_pairing_scheme *pairing)
{
	mtd->pairing = pairing;
}

static inline void mtd_set_of_node(struct mtd_info *mtd,
				   struct device_node *np)
{
	mtd->dev.of_node = np;
	if (!mtd->name)
		of_property_read_string(np, "label", &mtd->name);
}

static inline struct device_node *mtd_get_of_node(struct mtd_info *mtd)
{
	return dev_of_node(&mtd->dev);
}

static inline u32 mtd_oobavail(struct mtd_info *mtd, struct mtd_oob_ops *ops)
{
	return ops->mode == MTD_OPS_AUTO_OOB ? mtd->oobavail : mtd->oobsize;
}

static inline int mtd_max_bad_blocks(struct mtd_info *mtd,
				     loff_t ofs, size_t len)
{
	struct mtd_info *master = mtd_get_master(mtd);

	if (!master->_max_bad_blocks)
		return -ENOTSUPP;

	if (mtd->size < (len + ofs) || ofs < 0)
		return -EINVAL;

	return master->_max_bad_blocks(master, mtd_get_master_ofs(mtd, ofs),
				       len);
}

int mtd_wunit_to_pairing_info(struct mtd_info *mtd, int wunit,
			      struct mtd_pairing_info *info);
int mtd_pairing_info_to_wunit(struct mtd_info *mtd,
			      const struct mtd_pairing_info *info);
int mtd_pairing_groups(struct mtd_info *mtd);
int mtd_erase(struct mtd_info *mtd, struct erase_info *instr);
int mtd_point(struct mtd_info *mtd, loff_t from, size_t len, size_t *retlen,
	      void **virt, resource_size_t *phys);
int mtd_unpoint(struct mtd_info *mtd, loff_t from, size_t len);
unsigned long mtd_get_unmapped_area(struct mtd_info *mtd, unsigned long len,
				    unsigned long offset, unsigned long flags);
int mtd_read(struct mtd_info *mtd, loff_t from, size_t len, size_t *retlen,
	     u_char *buf);
int mtd_write(struct mtd_info *mtd, loff_t to, size_t len, size_t *retlen,
	      const u_char *buf);
int mtd_panic_write(struct mtd_info *mtd, loff_t to, size_t len, size_t *retlen,
		    const u_char *buf);

int mtd_read_oob(struct mtd_info *mtd, loff_t from, struct mtd_oob_ops *ops);
int mtd_write_oob(struct mtd_info *mtd, loff_t to, struct mtd_oob_ops *ops);

int mtd_get_fact_prot_info(struct mtd_info *mtd, size_t len, size_t *retlen,
			   struct otp_info *buf);
int mtd_read_fact_prot_reg(struct mtd_info *mtd, loff_t from, size_t len,
			   size_t *retlen, u_char *buf);
int mtd_get_user_prot_info(struct mtd_info *mtd, size_t len, size_t *retlen,
			   struct otp_info *buf);
int mtd_read_user_prot_reg(struct mtd_info *mtd, loff_t from, size_t len,
			   size_t *retlen, u_char *buf);
int mtd_write_user_prot_reg(struct mtd_info *mtd, loff_t to, size_t len,
			    size_t *retlen, const u_char *buf);
int mtd_lock_user_prot_reg(struct mtd_info *mtd, loff_t from, size_t len);
int mtd_erase_user_prot_reg(struct mtd_info *mtd, loff_t from, size_t len);

int mtd_writev(struct mtd_info *mtd, const struct kvec *vecs,
	       unsigned long count, loff_t to, size_t *retlen);

static inline void mtd_sync(struct mtd_info *mtd)
{
	struct mtd_info *master = mtd_get_master(mtd);

	if (master->_sync)
		master->_sync(master);
}

int mtd_lock(struct mtd_info *mtd, loff_t ofs, uint64_t len);
int mtd_unlock(struct mtd_info *mtd, loff_t ofs, uint64_t len);
int mtd_is_locked(struct mtd_info *mtd, loff_t ofs, uint64_t len);
int mtd_block_isreserved(struct mtd_info *mtd, loff_t ofs);
int mtd_block_isbad(struct mtd_info *mtd, loff_t ofs);
int mtd_block_markbad(struct mtd_info *mtd, loff_t ofs);

static inline int mtd_suspend(struct mtd_info *mtd)
{
	struct mtd_info *master = mtd_get_master(mtd);
	int ret;

	if (master->master.suspended)
		return 0;

	ret = master->_suspend ? master->_suspend(master) : 0;
	if (ret)
		return ret;

	master->master.suspended = 1;
	return 0;
}

static inline void mtd_resume(struct mtd_info *mtd)
{
	struct mtd_info *master = mtd_get_master(mtd);

	if (!master->master.suspended)
		return;

	if (master->_resume)
		master->_resume(master);

	master->master.suspended = 0;
}

static inline uint32_t mtd_div_by_eb(uint64_t sz, struct mtd_info *mtd)
{
	if (mtd->erasesize_shift)
		return sz >> mtd->erasesize_shift;
	do_div(sz, mtd->erasesize);
	return sz;
}

static inline uint32_t mtd_mod_by_eb(uint64_t sz, struct mtd_info *mtd)
{
	if (mtd->erasesize_shift)
		return sz & mtd->erasesize_mask;
	return do_div(sz, mtd->erasesize);
}


static inline void mtd_align_erase_req(struct mtd_info *mtd,
				       struct erase_info *req)
{
	u32 mod;

	if (WARN_ON(!mtd->erasesize))
		return;

	mod = mtd_mod_by_eb(req->addr, mtd);
	if (mod) {
		req->addr -= mod;
		req->len += mod;
	}

	mod = mtd_mod_by_eb(req->addr + req->len, mtd);
	if (mod)
		req->len += mtd->erasesize - mod;
}

static inline uint32_t mtd_div_by_ws(uint64_t sz, struct mtd_info *mtd)
{
	if (mtd->writesize_shift)
		return sz >> mtd->writesize_shift;
	do_div(sz, mtd->writesize);
	return sz;
}

static inline uint32_t mtd_mod_by_ws(uint64_t sz, struct mtd_info *mtd)
{
	if (mtd->writesize_shift)
		return sz & mtd->writesize_mask;
	return do_div(sz, mtd->writesize);
}

static inline int mtd_wunit_per_eb(struct mtd_info *mtd)
{
	struct mtd_info *master = mtd_get_master(mtd);

	return master->erasesize / mtd->writesize;
}

static inline int mtd_offset_to_wunit(struct mtd_info *mtd, loff_t offs)
{
	return mtd_div_by_ws(mtd_mod_by_eb(offs, mtd), mtd);
}

static inline loff_t mtd_wunit_to_offset(struct mtd_info *mtd, loff_t base,
					 int wunit)
{
	return base + (wunit * mtd->writesize);
}


static inline int mtd_has_oob(const struct mtd_info *mtd)
{
	struct mtd_info *master = mtd_get_master((struct mtd_info *)mtd);

	return master->_read_oob && master->_write_oob;
}

static inline int mtd_type_is_nand(const struct mtd_info *mtd)
{
	return mtd->type == MTD_NANDFLASH || mtd->type == MTD_MLCNANDFLASH;
}

static inline int mtd_can_have_bb(const struct mtd_info *mtd)
{
	struct mtd_info *master = mtd_get_master((struct mtd_info *)mtd);

	return !!master->_block_isbad;
}

	

struct mtd_partition;
struct mtd_part_parser_data;

extern int mtd_device_parse_register(struct mtd_info *mtd,
				     const char * const *part_probe_types,
				     struct mtd_part_parser_data *parser_data,
				     const struct mtd_partition *defparts,
				     int defnr_parts);
#define mtd_device_register(master, parts, nr_parts)	\
	mtd_device_parse_register(master, NULL, NULL, parts, nr_parts)
extern int mtd_device_unregister(struct mtd_info *master);
extern struct mtd_info *get_mtd_device(struct mtd_info *mtd, int num);
extern int __get_mtd_device(struct mtd_info *mtd);
extern void __put_mtd_device(struct mtd_info *mtd);
extern struct mtd_info *of_get_mtd_device_by_node(struct device_node *np);
extern struct mtd_info *get_mtd_device_nm(const char *name);
extern void put_mtd_device(struct mtd_info *mtd);


struct mtd_notifier {
	void (*add)(struct mtd_info *mtd);
	void (*remove)(struct mtd_info *mtd);
	struct list_head list;
};


extern void register_mtd_user (struct mtd_notifier *new);
extern int unregister_mtd_user (struct mtd_notifier *old);
void *mtd_kmalloc_up_to(const struct mtd_info *mtd, size_t *size);

static inline int mtd_is_bitflip(int err) {
	return err == -EUCLEAN;
}

static inline int mtd_is_eccerr(int err) {
	return err == -EBADMSG;
}

static inline int mtd_is_bitflip_or_eccerr(int err) {
	return mtd_is_bitflip(err) || mtd_is_eccerr(err);
}

unsigned mtd_mmap_capabilities(struct mtd_info *mtd);

#ifdef CONFIG_DEBUG_FS
bool mtd_check_expert_analysis_mode(void);
#else
static inline bool mtd_check_expert_analysis_mode(void) { return false; }
#endif


#endif 
