/* SPDX-License-Identifier: GPL-2.0 */


#ifndef _LINUX_NVMEM_PROVIDER_H
#define _LINUX_NVMEM_PROVIDER_H

#include <linux/device.h>
#include <linux/device/driver.h>
#include <linux/err.h>
#include <linux/errno.h>
#include <linux/gpio/consumer.h>

struct nvmem_device;
typedef int (*nvmem_reg_read_t)(void *priv, unsigned int offset,
				void *val, size_t bytes);
typedef int (*nvmem_reg_write_t)(void *priv, unsigned int offset,
				 void *val, size_t bytes);

typedef int (*nvmem_cell_post_process_t)(void *priv, const char *id, int index,
					 unsigned int offset, void *buf,
					 size_t bytes);

enum nvmem_type {
	NVMEM_TYPE_UNKNOWN = 0,
	NVMEM_TYPE_EEPROM,
	NVMEM_TYPE_OTP,
	NVMEM_TYPE_BATTERY_BACKED,
	NVMEM_TYPE_FRAM,
};

#define NVMEM_DEVID_NONE	(-1)
#define NVMEM_DEVID_AUTO	(-2)


struct nvmem_keepout {
	unsigned int start;
	unsigned int end;
	unsigned char value;
};


struct nvmem_cell_info {
	const char		*name;
	unsigned int		offset;
	size_t			raw_len;
	unsigned int		bytes;
	unsigned int		bit_offset;
	unsigned int		nbits;
	struct device_node	*np;
	nvmem_cell_post_process_t read_post_process;
	void			*priv;
};


struct nvmem_config {
	struct device		*dev;
	const char		*name;
	int			id;
	struct module		*owner;
	const struct nvmem_cell_info	*cells;
	int			ncells;
	bool			add_legacy_fixed_of_cells;
	void (*fixup_dt_cell_info)(struct nvmem_device *nvmem,
				   struct nvmem_cell_info *cell);
	const struct nvmem_keepout *keepout;
	unsigned int		nkeepout;
	enum nvmem_type		type;
	bool			read_only;
	bool			root_only;
	bool			ignore_wp;
	struct nvmem_layout	*layout;
	struct device_node	*of_node;
	nvmem_reg_read_t	reg_read;
	nvmem_reg_write_t	reg_write;
	int	size;
	int	word_size;
	int	stride;
	void	*priv;
	
	bool			compat;
	struct device		*base_dev;
};


struct nvmem_cell_table {
	const char		*nvmem_name;
	const struct nvmem_cell_info	*cells;
	size_t			ncells;
	struct list_head	node;
};


struct nvmem_layout {
	struct device dev;
	struct nvmem_device *nvmem;
	int (*add_cells)(struct nvmem_layout *layout);
};

struct nvmem_layout_driver {
	struct device_driver driver;
	int (*probe)(struct nvmem_layout *layout);
	void (*remove)(struct nvmem_layout *layout);
};

#if IS_ENABLED(CONFIG_NVMEM)

struct nvmem_device *nvmem_register(const struct nvmem_config *cfg);
void nvmem_unregister(struct nvmem_device *nvmem);

struct nvmem_device *devm_nvmem_register(struct device *dev,
					 const struct nvmem_config *cfg);

void nvmem_add_cell_table(struct nvmem_cell_table *table);
void nvmem_del_cell_table(struct nvmem_cell_table *table);

int nvmem_add_one_cell(struct nvmem_device *nvmem,
		       const struct nvmem_cell_info *info);

int nvmem_layout_register(struct nvmem_layout *layout);
void nvmem_layout_unregister(struct nvmem_layout *layout);

#define nvmem_layout_driver_register(drv) \
	__nvmem_layout_driver_register(drv, THIS_MODULE)
int __nvmem_layout_driver_register(struct nvmem_layout_driver *drv,
				   struct module *owner);
void nvmem_layout_driver_unregister(struct nvmem_layout_driver *drv);
#define module_nvmem_layout_driver(__nvmem_layout_driver)		\
	module_driver(__nvmem_layout_driver, nvmem_layout_driver_register, \
		      nvmem_layout_driver_unregister)

#else

static inline struct nvmem_device *nvmem_register(const struct nvmem_config *c)
{
	return ERR_PTR(-EOPNOTSUPP);
}

static inline void nvmem_unregister(struct nvmem_device *nvmem) {}

static inline struct nvmem_device *
devm_nvmem_register(struct device *dev, const struct nvmem_config *c)
{
	return nvmem_register(c);
}

static inline void nvmem_add_cell_table(struct nvmem_cell_table *table) {}
static inline void nvmem_del_cell_table(struct nvmem_cell_table *table) {}
static inline int nvmem_add_one_cell(struct nvmem_device *nvmem,
				     const struct nvmem_cell_info *info)
{
	return -EOPNOTSUPP;
}

static inline int nvmem_layout_register(struct nvmem_layout *layout)
{
	return -EOPNOTSUPP;
}

static inline void nvmem_layout_unregister(struct nvmem_layout *layout) {}

#endif 

#if IS_ENABLED(CONFIG_NVMEM) && IS_ENABLED(CONFIG_OF)


struct device_node *of_nvmem_layout_get_container(struct nvmem_device *nvmem);

#else  

static inline struct device_node *of_nvmem_layout_get_container(struct nvmem_device *nvmem)
{
	return NULL;
}

#endif 

#endif  
