/* SPDX-License-Identifier: GPL-2.0 */


#ifndef _RISCV_PMU_H
#define _RISCV_PMU_H

#include <linux/perf_event.h>
#include <linux/ptrace.h>
#include <linux/interrupt.h>

#ifdef CONFIG_RISCV_PMU



#define RISCV_MAX_COUNTERS	64
#define RISCV_OP_UNSUPP		(-EOPNOTSUPP)
#define RISCV_PMU_SBI_PDEV_NAME	"riscv-pmu-sbi"
#define RISCV_PMU_LEGACY_PDEV_NAME	"riscv-pmu-legacy"

#define RISCV_PMU_STOP_FLAG_RESET 1

#define RISCV_PMU_CONFIG1_GUEST_EVENTS 0x1

struct cpu_hw_events {
	
	int			n_events;
	
	int		irq;
	
	struct perf_event	*events[RISCV_MAX_COUNTERS];
	
	DECLARE_BITMAP(used_hw_ctrs, RISCV_MAX_COUNTERS);
	
	DECLARE_BITMAP(used_fw_ctrs, RISCV_MAX_COUNTERS);
	
	void *snapshot_addr;
	
	phys_addr_t snapshot_addr_phys;
	
	bool snapshot_set_done;
	
	u64 snapshot_cval_shcopy[RISCV_MAX_COUNTERS];
};

struct riscv_pmu {
	struct pmu	pmu;
	char		*name;

	irqreturn_t	(*handle_irq)(int irq_num, void *dev);

	unsigned long	cmask;
	u64		(*ctr_read)(struct perf_event *event);
	int		(*ctr_get_idx)(struct perf_event *event);
	int		(*ctr_get_width)(int idx);
	void		(*ctr_clear_idx)(struct perf_event *event);
	void		(*ctr_start)(struct perf_event *event, u64 init_val);
	void		(*ctr_stop)(struct perf_event *event, unsigned long flag);
	int		(*event_map)(struct perf_event *event, u64 *config);
	void		(*event_init)(struct perf_event *event);
	void		(*event_mapped)(struct perf_event *event, struct mm_struct *mm);
	void		(*event_unmapped)(struct perf_event *event, struct mm_struct *mm);
	uint8_t		(*csr_index)(struct perf_event *event);

	struct cpu_hw_events	__percpu *hw_events;
	struct hlist_node	node;
	struct notifier_block   riscv_pm_nb;
};

#define to_riscv_pmu(p) (container_of(p, struct riscv_pmu, pmu))

void riscv_pmu_start(struct perf_event *event, int flags);
void riscv_pmu_stop(struct perf_event *event, int flags);
unsigned long riscv_pmu_ctr_read_csr(unsigned long csr);
int riscv_pmu_event_set_period(struct perf_event *event);
uint64_t riscv_pmu_ctr_get_width_mask(struct perf_event *event);
u64 riscv_pmu_event_update(struct perf_event *event);
#ifdef CONFIG_RISCV_PMU_LEGACY
void riscv_pmu_legacy_skip_init(void);
#else
static inline void riscv_pmu_legacy_skip_init(void) {};
#endif
struct riscv_pmu *riscv_pmu_alloc(void);
#ifdef CONFIG_RISCV_PMU_SBI
int riscv_pmu_get_hpm_info(u32 *hw_ctr_width, u32 *num_hw_ctr);
#endif

#endif 

#endif 
