/* SPDX-License-Identifier: GPL-2.0-only */


#ifndef __LINUX_REGULATOR_CONSUMER_H_
#define __LINUX_REGULATOR_CONSUMER_H_

#include <linux/err.h>
#include <linux/suspend.h>
#include <regulator/regulator.h>

struct device;
struct notifier_block;
struct regmap;
struct regulator_dev;



#define REGULATOR_MODE_INVALID			0x0
#define REGULATOR_MODE_FAST			0x1
#define REGULATOR_MODE_NORMAL			0x2
#define REGULATOR_MODE_IDLE			0x4
#define REGULATOR_MODE_STANDBY			0x8



#define REGULATOR_ERROR_UNDER_VOLTAGE		BIT(1)
#define REGULATOR_ERROR_OVER_CURRENT		BIT(2)
#define REGULATOR_ERROR_REGULATION_OUT		BIT(3)
#define REGULATOR_ERROR_FAIL			BIT(4)
#define REGULATOR_ERROR_OVER_TEMP		BIT(5)

#define REGULATOR_ERROR_UNDER_VOLTAGE_WARN	BIT(6)
#define REGULATOR_ERROR_OVER_CURRENT_WARN	BIT(7)
#define REGULATOR_ERROR_OVER_VOLTAGE_WARN	BIT(8)
#define REGULATOR_ERROR_OVER_TEMP_WARN		BIT(9)


struct pre_voltage_change_data {
	unsigned long old_uV;
	unsigned long min_uV;
	unsigned long max_uV;
};

struct regulator;


struct regulator_bulk_data {
	const char *supply;
	struct regulator *consumer;
	int init_load_uA;

	
	int ret;
};

#if defined(CONFIG_REGULATOR)


struct regulator *__must_check regulator_get(struct device *dev,
					     const char *id);
struct regulator *__must_check devm_regulator_get(struct device *dev,
					     const char *id);
struct regulator *__must_check regulator_get_exclusive(struct device *dev,
						       const char *id);
struct regulator *__must_check devm_regulator_get_exclusive(struct device *dev,
							const char *id);
struct regulator *__must_check regulator_get_optional(struct device *dev,
						      const char *id);
struct regulator *__must_check devm_regulator_get_optional(struct device *dev,
							   const char *id);
int devm_regulator_get_enable(struct device *dev, const char *id);
int devm_regulator_get_enable_optional(struct device *dev, const char *id);
int devm_regulator_get_enable_read_voltage(struct device *dev, const char *id);
void regulator_put(struct regulator *regulator);
void devm_regulator_put(struct regulator *regulator);

int regulator_register_supply_alias(struct device *dev, const char *id,
				    struct device *alias_dev,
				    const char *alias_id);
void regulator_unregister_supply_alias(struct device *dev, const char *id);

int regulator_bulk_register_supply_alias(struct device *dev,
					 const char *const *id,
					 struct device *alias_dev,
					 const char *const *alias_id,
					 int num_id);
void regulator_bulk_unregister_supply_alias(struct device *dev,
					    const char * const *id, int num_id);

int devm_regulator_register_supply_alias(struct device *dev, const char *id,
					 struct device *alias_dev,
					 const char *alias_id);

int devm_regulator_bulk_register_supply_alias(struct device *dev,
					      const char *const *id,
					      struct device *alias_dev,
					      const char *const *alias_id,
					      int num_id);


int __must_check regulator_enable(struct regulator *regulator);
int regulator_disable(struct regulator *regulator);
int regulator_force_disable(struct regulator *regulator);
int regulator_is_enabled(struct regulator *regulator);
int regulator_disable_deferred(struct regulator *regulator, int ms);

int __must_check regulator_bulk_get(struct device *dev, int num_consumers,
				    struct regulator_bulk_data *consumers);
int __must_check devm_regulator_bulk_get(struct device *dev, int num_consumers,
					 struct regulator_bulk_data *consumers);
void devm_regulator_bulk_put(struct regulator_bulk_data *consumers);
int __must_check devm_regulator_bulk_get_exclusive(struct device *dev, int num_consumers,
						   struct regulator_bulk_data *consumers);
int __must_check devm_regulator_bulk_get_const(
	struct device *dev, int num_consumers,
	const struct regulator_bulk_data *in_consumers,
	struct regulator_bulk_data **out_consumers);
int __must_check regulator_bulk_enable(int num_consumers,
				       struct regulator_bulk_data *consumers);
int devm_regulator_bulk_get_enable(struct device *dev, int num_consumers,
				   const char * const *id);
int regulator_bulk_disable(int num_consumers,
			   struct regulator_bulk_data *consumers);
int regulator_bulk_force_disable(int num_consumers,
			   struct regulator_bulk_data *consumers);
void regulator_bulk_free(int num_consumers,
			 struct regulator_bulk_data *consumers);

int regulator_count_voltages(struct regulator *regulator);
int regulator_list_voltage(struct regulator *regulator, unsigned selector);
int regulator_is_supported_voltage(struct regulator *regulator,
				   int min_uV, int max_uV);
unsigned int regulator_get_linear_step(struct regulator *regulator);
int regulator_set_voltage(struct regulator *regulator, int min_uV, int max_uV);
int regulator_set_voltage_time(struct regulator *regulator,
			       int old_uV, int new_uV);
int regulator_get_voltage(struct regulator *regulator);
int regulator_sync_voltage(struct regulator *regulator);
int regulator_set_current_limit(struct regulator *regulator,
			       int min_uA, int max_uA);
int regulator_get_current_limit(struct regulator *regulator);

int regulator_set_mode(struct regulator *regulator, unsigned int mode);
unsigned int regulator_get_mode(struct regulator *regulator);
int regulator_get_error_flags(struct regulator *regulator,
				unsigned int *flags);
int regulator_set_load(struct regulator *regulator, int load_uA);

int regulator_allow_bypass(struct regulator *regulator, bool allow);

struct regmap *regulator_get_regmap(struct regulator *regulator);
int regulator_get_hardware_vsel_register(struct regulator *regulator,
					 unsigned *vsel_reg,
					 unsigned *vsel_mask);
int regulator_list_hardware_vsel(struct regulator *regulator,
				 unsigned selector);
int regulator_hardware_enable(struct regulator *regulator, bool enable);


int regulator_register_notifier(struct regulator *regulator,
			      struct notifier_block *nb);
int devm_regulator_register_notifier(struct regulator *regulator,
				     struct notifier_block *nb);
int regulator_unregister_notifier(struct regulator *regulator,
				struct notifier_block *nb);
void devm_regulator_unregister_notifier(struct regulator *regulator,
					struct notifier_block *nb);


int regulator_suspend_enable(struct regulator_dev *rdev,
			     suspend_state_t state);
int regulator_suspend_disable(struct regulator_dev *rdev,
			      suspend_state_t state);
int regulator_set_suspend_voltage(struct regulator *regulator, int min_uV,
				  int max_uV, suspend_state_t state);


void *regulator_get_drvdata(struct regulator *regulator);
void regulator_set_drvdata(struct regulator *regulator, void *data);



void regulator_bulk_set_supply_names(struct regulator_bulk_data *consumers,
				     const char *const *supply_names,
				     unsigned int num_supplies);

bool regulator_is_equal(struct regulator *reg1, struct regulator *reg2);

#else


static inline struct regulator *__must_check regulator_get(struct device *dev,
	const char *id)
{
	
	return NULL;
}

static inline struct regulator *__must_check
devm_regulator_get(struct device *dev, const char *id)
{
	return NULL;
}

static inline struct regulator *__must_check
regulator_get_exclusive(struct device *dev, const char *id)
{
	return ERR_PTR(-ENODEV);
}

static inline struct regulator *__must_check
devm_regulator_get_exclusive(struct device *dev, const char *id)
{
	return ERR_PTR(-ENODEV);
}

static inline int devm_regulator_get_enable(struct device *dev, const char *id)
{
	return 0;
}

static inline int devm_regulator_get_enable_optional(struct device *dev,
						     const char *id)
{
	return 0;
}

static inline int devm_regulator_get_enable_read_voltage(struct device *dev,
							 const char *id)
{
	return -ENODEV;
}

static inline struct regulator *__must_check
regulator_get_optional(struct device *dev, const char *id)
{
	return ERR_PTR(-ENODEV);
}


static inline struct regulator *__must_check
devm_regulator_get_optional(struct device *dev, const char *id)
{
	return ERR_PTR(-ENODEV);
}

static inline void regulator_put(struct regulator *regulator)
{
}

static inline void devm_regulator_put(struct regulator *regulator)
{
}

static inline void devm_regulator_bulk_put(struct regulator_bulk_data *consumers)
{
}

static inline int regulator_register_supply_alias(struct device *dev,
						  const char *id,
						  struct device *alias_dev,
						  const char *alias_id)
{
	return 0;
}

static inline void regulator_unregister_supply_alias(struct device *dev,
						    const char *id)
{
}

static inline int regulator_bulk_register_supply_alias(struct device *dev,
						const char *const *id,
						struct device *alias_dev,
						const char * const *alias_id,
						int num_id)
{
	return 0;
}

static inline void regulator_bulk_unregister_supply_alias(struct device *dev,
						const char * const *id,
						int num_id)
{
}

static inline int devm_regulator_register_supply_alias(struct device *dev,
						       const char *id,
						       struct device *alias_dev,
						       const char *alias_id)
{
	return 0;
}

static inline int devm_regulator_bulk_register_supply_alias(struct device *dev,
						const char *const *id,
						struct device *alias_dev,
						const char *const *alias_id,
						int num_id)
{
	return 0;
}

static inline int regulator_enable(struct regulator *regulator)
{
	return 0;
}

static inline int regulator_disable(struct regulator *regulator)
{
	return 0;
}

static inline int regulator_force_disable(struct regulator *regulator)
{
	return 0;
}

static inline int regulator_disable_deferred(struct regulator *regulator,
					     int ms)
{
	return 0;
}

static inline int regulator_is_enabled(struct regulator *regulator)
{
	return 1;
}

static inline int regulator_bulk_get(struct device *dev,
				     int num_consumers,
				     struct regulator_bulk_data *consumers)
{
	return 0;
}

static inline int devm_regulator_bulk_get(struct device *dev, int num_consumers,
					  struct regulator_bulk_data *consumers)
{
	return 0;
}

static inline int devm_regulator_bulk_get_const(
	struct device *dev, int num_consumers,
	const struct regulator_bulk_data *in_consumers,
	struct regulator_bulk_data **out_consumers)
{
	return 0;
}

static inline int regulator_bulk_enable(int num_consumers,
					struct regulator_bulk_data *consumers)
{
	return 0;
}

static inline int devm_regulator_bulk_get_enable(struct device *dev,
						 int num_consumers,
						 const char * const *id)
{
	return 0;
}

static inline int regulator_bulk_disable(int num_consumers,
					 struct regulator_bulk_data *consumers)
{
	return 0;
}

static inline int regulator_bulk_force_disable(int num_consumers,
					struct regulator_bulk_data *consumers)
{
	return 0;
}

static inline void regulator_bulk_free(int num_consumers,
				       struct regulator_bulk_data *consumers)
{
}

static inline int regulator_set_voltage(struct regulator *regulator,
					int min_uV, int max_uV)
{
	return 0;
}

static inline int regulator_set_voltage_time(struct regulator *regulator,
					     int old_uV, int new_uV)
{
	return 0;
}

static inline int regulator_get_voltage(struct regulator *regulator)
{
	return -EINVAL;
}

static inline int regulator_sync_voltage(struct regulator *regulator)
{
	return -EINVAL;
}

static inline int regulator_is_supported_voltage(struct regulator *regulator,
				   int min_uV, int max_uV)
{
	return 0;
}

static inline unsigned int regulator_get_linear_step(struct regulator *regulator)
{
	return 0;
}

static inline int regulator_set_current_limit(struct regulator *regulator,
					     int min_uA, int max_uA)
{
	return 0;
}

static inline int regulator_get_current_limit(struct regulator *regulator)
{
	return 0;
}

static inline int regulator_set_mode(struct regulator *regulator,
	unsigned int mode)
{
	return 0;
}

static inline unsigned int regulator_get_mode(struct regulator *regulator)
{
	return REGULATOR_MODE_NORMAL;
}

static inline int regulator_get_error_flags(struct regulator *regulator,
					    unsigned int *flags)
{
	return -EINVAL;
}

static inline int regulator_set_load(struct regulator *regulator, int load_uA)
{
	return 0;
}

static inline int regulator_allow_bypass(struct regulator *regulator,
					 bool allow)
{
	return 0;
}

static inline struct regmap *regulator_get_regmap(struct regulator *regulator)
{
	return ERR_PTR(-EOPNOTSUPP);
}

static inline int regulator_get_hardware_vsel_register(struct regulator *regulator,
						       unsigned *vsel_reg,
						       unsigned *vsel_mask)
{
	return -EOPNOTSUPP;
}

static inline int regulator_list_hardware_vsel(struct regulator *regulator,
					       unsigned selector)
{
	return -EOPNOTSUPP;
}

static inline int regulator_hardware_enable(struct regulator *regulator,
					    bool enable)
{
	return -EOPNOTSUPP;
}

static inline int regulator_register_notifier(struct regulator *regulator,
			      struct notifier_block *nb)
{
	return 0;
}

static inline int devm_regulator_register_notifier(struct regulator *regulator,
						   struct notifier_block *nb)
{
	return 0;
}

static inline int regulator_unregister_notifier(struct regulator *regulator,
				struct notifier_block *nb)
{
	return 0;
}

static inline int devm_regulator_unregister_notifier(struct regulator *regulator,
						     struct notifier_block *nb)
{
	return 0;
}

static inline int regulator_suspend_enable(struct regulator_dev *rdev,
					   suspend_state_t state)
{
	return -EINVAL;
}

static inline int regulator_suspend_disable(struct regulator_dev *rdev,
					    suspend_state_t state)
{
	return -EINVAL;
}

static inline int regulator_set_suspend_voltage(struct regulator *regulator,
						int min_uV, int max_uV,
						suspend_state_t state)
{
	return -EINVAL;
}

static inline void *regulator_get_drvdata(struct regulator *regulator)
{
	return NULL;
}

static inline void regulator_set_drvdata(struct regulator *regulator,
	void *data)
{
}

static inline int regulator_count_voltages(struct regulator *regulator)
{
	return 0;
}

static inline int regulator_list_voltage(struct regulator *regulator, unsigned selector)
{
	return -EINVAL;
}

static inline void
regulator_bulk_set_supply_names(struct regulator_bulk_data *consumers,
				const char *const *supply_names,
				unsigned int num_supplies)
{
}

static inline bool
regulator_is_equal(struct regulator *reg1, struct regulator *reg2)
{
	return false;
}
#endif

#if IS_ENABLED(CONFIG_OF) && IS_ENABLED(CONFIG_REGULATOR)
struct regulator *__must_check of_regulator_get_optional(struct device *dev,
							 struct device_node *node,
							 const char *id);
struct regulator *__must_check devm_of_regulator_get_optional(struct device *dev,
							      struct device_node *node,
							      const char *id);
int __must_check of_regulator_bulk_get_all(struct device *dev, struct device_node *np,
					   struct regulator_bulk_data **consumers);
#else
static inline struct regulator *__must_check of_regulator_get_optional(struct device *dev,
								       struct device_node *node,
								       const char *id)
{
	return ERR_PTR(-ENODEV);
}

static inline struct regulator *__must_check devm_of_regulator_get_optional(struct device *dev,
									    struct device_node *node,
									    const char *id)
{
	return ERR_PTR(-ENODEV);
}

static inline int of_regulator_bulk_get_all(struct device *dev, struct device_node *np,
					    struct regulator_bulk_data **consumers)
{
	return 0;
}

#endif

static inline int regulator_set_voltage_triplet(struct regulator *regulator,
						int min_uV, int target_uV,
						int max_uV)
{
	if (regulator_set_voltage(regulator, target_uV, max_uV) == 0)
		return 0;

	return regulator_set_voltage(regulator, min_uV, max_uV);
}

static inline int regulator_set_voltage_tol(struct regulator *regulator,
					    int new_uV, int tol_uV)
{
	if (regulator_set_voltage(regulator, new_uV, new_uV + tol_uV) == 0)
		return 0;
	else
		return regulator_set_voltage(regulator,
					     new_uV - tol_uV, new_uV + tol_uV);
}

static inline int regulator_is_supported_voltage_tol(struct regulator *regulator,
						     int target_uV, int tol_uV)
{
	return regulator_is_supported_voltage(regulator,
					      target_uV - tol_uV,
					      target_uV + tol_uV);
}

#endif
