/* SPDX-License-Identifier: GPL-2.0-only */


#ifndef __LINUX_REGULATOR_DRIVER_H_
#define __LINUX_REGULATOR_DRIVER_H_

#include <linux/device.h>
#include <linux/linear_range.h>
#include <linux/notifier.h>
#include <linux/regulator/consumer.h>
#include <linux/ww_mutex.h>

struct gpio_desc;
struct regmap;
struct regulator_dev;
struct regulator_config;
struct regulator_init_data;
struct regulator_enable_gpio;

enum regulator_status {
	REGULATOR_STATUS_OFF,
	REGULATOR_STATUS_ON,
	REGULATOR_STATUS_ERROR,
	
	REGULATOR_STATUS_FAST,
	REGULATOR_STATUS_NORMAL,
	REGULATOR_STATUS_IDLE,
	REGULATOR_STATUS_STANDBY,
	
	REGULATOR_STATUS_BYPASS,
	
	REGULATOR_STATUS_UNDEFINED,
};

enum regulator_detection_severity {
	
	REGULATOR_SEVERITY_PROT,
	
	REGULATOR_SEVERITY_ERR,
	
	REGULATOR_SEVERITY_WARN,
};


#define REGULATOR_LINEAR_RANGE(_min_uV, _min_sel, _max_sel, _step_uV)	\
	LINEAR_RANGE(_min_uV, _min_sel, _max_sel, _step_uV)


struct regulator_ops {

	
	int (*list_voltage) (struct regulator_dev *, unsigned selector);

	
	int (*set_voltage) (struct regulator_dev *, int min_uV, int max_uV,
			    unsigned *selector);
	int (*map_voltage)(struct regulator_dev *, int min_uV, int max_uV);
	int (*set_voltage_sel) (struct regulator_dev *, unsigned selector);
	int (*get_voltage) (struct regulator_dev *);
	int (*get_voltage_sel) (struct regulator_dev *);

	
	int (*set_current_limit) (struct regulator_dev *,
				 int min_uA, int max_uA);
	int (*get_current_limit) (struct regulator_dev *);

	int (*set_input_current_limit) (struct regulator_dev *, int lim_uA);
	int (*set_over_current_protection)(struct regulator_dev *, int lim_uA,
					   int severity, bool enable);
	int (*set_over_voltage_protection)(struct regulator_dev *, int lim_uV,
					   int severity, bool enable);
	int (*set_under_voltage_protection)(struct regulator_dev *, int lim_uV,
					    int severity, bool enable);
	int (*set_thermal_protection)(struct regulator_dev *, int lim,
				      int severity, bool enable);
	int (*set_active_discharge)(struct regulator_dev *, bool enable);

	
	int (*enable) (struct regulator_dev *);
	int (*disable) (struct regulator_dev *);
	int (*is_enabled) (struct regulator_dev *);

	
	int (*set_mode) (struct regulator_dev *, unsigned int mode);
	unsigned int (*get_mode) (struct regulator_dev *);

	
	int (*get_error_flags)(struct regulator_dev *, unsigned int *flags);

	
	int (*enable_time) (struct regulator_dev *);
	int (*set_ramp_delay) (struct regulator_dev *, int ramp_delay);
	int (*set_voltage_time) (struct regulator_dev *, int old_uV,
				 int new_uV);
	int (*set_voltage_time_sel) (struct regulator_dev *,
				     unsigned int old_selector,
				     unsigned int new_selector);

	int (*set_soft_start) (struct regulator_dev *);

	
	int (*get_status)(struct regulator_dev *);

	
	unsigned int (*get_optimum_mode) (struct regulator_dev *, int input_uV,
					  int output_uV, int load_uA);
	
	int (*set_load)(struct regulator_dev *, int load_uA);

	
	int (*set_bypass)(struct regulator_dev *dev, bool enable);
	int (*get_bypass)(struct regulator_dev *dev, bool *enable);

	

	
	int (*set_suspend_voltage) (struct regulator_dev *, int uV);

	
	int (*set_suspend_enable) (struct regulator_dev *);
	int (*set_suspend_disable) (struct regulator_dev *);

	
	int (*set_suspend_mode) (struct regulator_dev *, unsigned int mode);

	int (*resume)(struct regulator_dev *rdev);

	int (*set_pull_down) (struct regulator_dev *);
};


enum regulator_type {
	REGULATOR_VOLTAGE,
	REGULATOR_CURRENT,
};


struct regulator_desc {
	const char *name;
	const char *supply_name;
	const char *of_match;
	bool of_match_full_name;
	const char *regulators_node;
	int (*of_parse_cb)(struct device_node *,
			    const struct regulator_desc *,
			    struct regulator_config *);
	int (*init_cb)(struct regulator_dev *,
		       struct regulator_config *);
	int id;
	unsigned int continuous_voltage_range:1;
	unsigned n_voltages;
	unsigned int n_current_limits;
	const struct regulator_ops *ops;
	int irq;
	enum regulator_type type;
	struct module *owner;

	unsigned int min_uV;
	unsigned int uV_step;
	unsigned int linear_min_sel;
	int fixed_uV;
	unsigned int ramp_delay;
	int min_dropout_uV;

	const struct linear_range *linear_ranges;
	const unsigned int *linear_range_selectors_bitfield;

	int n_linear_ranges;

	const unsigned int *volt_table;
	const unsigned int *curr_table;

	unsigned int vsel_range_reg;
	unsigned int vsel_range_mask;
	bool range_applied_by_vsel;
	unsigned int vsel_reg;
	unsigned int vsel_mask;
	unsigned int vsel_step;
	unsigned int csel_reg;
	unsigned int csel_mask;
	unsigned int apply_reg;
	unsigned int apply_bit;
	unsigned int enable_reg;
	unsigned int enable_mask;
	unsigned int enable_val;
	unsigned int disable_val;
	bool enable_is_inverted;
	unsigned int bypass_reg;
	unsigned int bypass_mask;
	unsigned int bypass_val_on;
	unsigned int bypass_val_off;
	unsigned int active_discharge_on;
	unsigned int active_discharge_off;
	unsigned int active_discharge_mask;
	unsigned int active_discharge_reg;
	unsigned int soft_start_reg;
	unsigned int soft_start_mask;
	unsigned int soft_start_val_on;
	unsigned int pull_down_reg;
	unsigned int pull_down_mask;
	unsigned int pull_down_val_on;
	unsigned int ramp_reg;
	unsigned int ramp_mask;
	const unsigned int *ramp_delay_table;
	unsigned int n_ramp_values;

	unsigned int enable_time;

	unsigned int off_on_delay;

	unsigned int poll_enabled_time;

	unsigned int (*of_map_mode)(unsigned int mode);
};


struct regulator_config {
	struct device *dev;
	const struct regulator_init_data *init_data;
	void *driver_data;
	struct device_node *of_node;
	struct regmap *regmap;

	struct gpio_desc *ena_gpiod;
};


struct regulator_err_state {
	struct regulator_dev *rdev;
	unsigned long notifs;
	unsigned long errors;
	int possible_errs;
};


struct regulator_irq_data {
	struct regulator_err_state *states;
	int num_states;
	void *data;
	long opaque;
};


struct regulator_irq_desc {
	const char *name;
	int fatal_cnt;
	int reread_ms;
	int irq_off_ms;
	bool skip_off;
	bool high_prio;
	void *data;

	int (*die)(struct regulator_irq_data *rid);
	int (*map_event)(int irq, struct regulator_irq_data *rid,
			  unsigned long *dev_mask);
	int (*renable)(struct regulator_irq_data *rid);
};


enum {
	REGULATOR_ERROR_CLEARED,
	REGULATOR_FAILED_RETRY,
	REGULATOR_ERROR_ON,
};


struct coupling_desc {
	struct regulator_dev **coupled_rdevs;
	struct regulator_coupler *coupler;
	int n_resolved;
	int n_coupled;
};


struct regulator_dev {
	const struct regulator_desc *desc;
	int exclusive;
	u32 use_count;
	u32 open_count;
	u32 bypass_count;

	
	struct list_head list; 

	
	struct list_head consumer_list; 

	struct coupling_desc coupling_desc;

	struct blocking_notifier_head notifier;
	struct ww_mutex mutex; 
	struct task_struct *mutex_owner;
	int ref_cnt;
	struct module *owner;
	struct device dev;
	struct regulation_constraints *constraints;
	struct regulator *supply;	
	const char *supply_name;
	struct regmap *regmap;

	struct delayed_work disable_work;

	void *reg_data;		

	struct dentry *debugfs;

	struct regulator_enable_gpio *ena_pin;
	unsigned int ena_gpio_state:1;

	unsigned int is_switch:1;

	
	ktime_t last_off;
	int cached_err;
	bool use_cached_err;
	spinlock_t err_lock;
};


static inline int regulator_err2notif(int err)
{
	switch (err) {
	case REGULATOR_ERROR_UNDER_VOLTAGE:
		return REGULATOR_EVENT_UNDER_VOLTAGE;
	case REGULATOR_ERROR_OVER_CURRENT:
		return REGULATOR_EVENT_OVER_CURRENT;
	case REGULATOR_ERROR_REGULATION_OUT:
		return REGULATOR_EVENT_REGULATION_OUT;
	case REGULATOR_ERROR_FAIL:
		return REGULATOR_EVENT_FAIL;
	case REGULATOR_ERROR_OVER_TEMP:
		return REGULATOR_EVENT_OVER_TEMP;
	case REGULATOR_ERROR_UNDER_VOLTAGE_WARN:
		return REGULATOR_EVENT_UNDER_VOLTAGE_WARN;
	case REGULATOR_ERROR_OVER_CURRENT_WARN:
		return REGULATOR_EVENT_OVER_CURRENT_WARN;
	case REGULATOR_ERROR_OVER_VOLTAGE_WARN:
		return REGULATOR_EVENT_OVER_VOLTAGE_WARN;
	case REGULATOR_ERROR_OVER_TEMP_WARN:
		return REGULATOR_EVENT_OVER_TEMP_WARN;
	}
	return 0;
}


struct regulator_dev *
regulator_register(struct device *dev,
		   const struct regulator_desc *regulator_desc,
		   const struct regulator_config *config);
struct regulator_dev *
devm_regulator_register(struct device *dev,
			const struct regulator_desc *regulator_desc,
			const struct regulator_config *config);
void regulator_unregister(struct regulator_dev *rdev);

int regulator_notifier_call_chain(struct regulator_dev *rdev,
				  unsigned long event, void *data);
void *devm_regulator_irq_helper(struct device *dev,
				const struct regulator_irq_desc *d, int irq,
				int irq_flags, int common_errs,
				int *per_rdev_errs, struct regulator_dev **rdev,
				int rdev_amount);
void *regulator_irq_helper(struct device *dev,
			   const struct regulator_irq_desc *d, int irq,
			   int irq_flags, int common_errs, int *per_rdev_errs,
			   struct regulator_dev **rdev, int rdev_amount);
void regulator_irq_helper_cancel(void **handle);
int regulator_irq_map_event_simple(int irq, struct regulator_irq_data *rid,
				   unsigned long *dev_mask);

void *rdev_get_drvdata(struct regulator_dev *rdev);
struct device *rdev_get_dev(struct regulator_dev *rdev);
struct regmap *rdev_get_regmap(struct regulator_dev *rdev);
int rdev_get_id(struct regulator_dev *rdev);

int regulator_mode_to_status(unsigned int);

int regulator_list_voltage_linear(struct regulator_dev *rdev,
				  unsigned int selector);
int regulator_list_voltage_pickable_linear_range(struct regulator_dev *rdev,
						   unsigned int selector);
int regulator_list_voltage_linear_range(struct regulator_dev *rdev,
					unsigned int selector);
int regulator_list_voltage_table(struct regulator_dev *rdev,
				  unsigned int selector);
int regulator_map_voltage_linear(struct regulator_dev *rdev,
				  int min_uV, int max_uV);
int regulator_map_voltage_pickable_linear_range(struct regulator_dev *rdev,
						  int min_uV, int max_uV);
int regulator_map_voltage_linear_range(struct regulator_dev *rdev,
				       int min_uV, int max_uV);
int regulator_map_voltage_iterate(struct regulator_dev *rdev,
				  int min_uV, int max_uV);
int regulator_map_voltage_ascend(struct regulator_dev *rdev,
				  int min_uV, int max_uV);
int regulator_get_voltage_sel_pickable_regmap(struct regulator_dev *rdev);
int regulator_set_voltage_sel_pickable_regmap(struct regulator_dev *rdev,
						unsigned int sel);
int regulator_get_voltage_sel_regmap(struct regulator_dev *rdev);
int regulator_set_voltage_sel_regmap(struct regulator_dev *rdev, unsigned sel);
int regulator_is_enabled_regmap(struct regulator_dev *rdev);
int regulator_enable_regmap(struct regulator_dev *rdev);
int regulator_disable_regmap(struct regulator_dev *rdev);
int regulator_set_voltage_time_sel(struct regulator_dev *rdev,
				   unsigned int old_selector,
				   unsigned int new_selector);
int regulator_set_bypass_regmap(struct regulator_dev *rdev, bool enable);
int regulator_get_bypass_regmap(struct regulator_dev *rdev, bool *enable);
int regulator_set_soft_start_regmap(struct regulator_dev *rdev);
int regulator_set_pull_down_regmap(struct regulator_dev *rdev);

int regulator_set_active_discharge_regmap(struct regulator_dev *rdev,
					  bool enable);
int regulator_set_current_limit_regmap(struct regulator_dev *rdev,
				       int min_uA, int max_uA);
int regulator_get_current_limit_regmap(struct regulator_dev *rdev);
void *regulator_get_init_drvdata(struct regulator_init_data *reg_init_data);
int regulator_find_closest_bigger(unsigned int target, const unsigned int *table,
				  unsigned int num_sel, unsigned int *sel);
int regulator_set_ramp_delay_regmap(struct regulator_dev *rdev, int ramp_delay);
int regulator_sync_voltage_rdev(struct regulator_dev *rdev);


int regulator_desc_list_voltage_linear_range(const struct regulator_desc *desc,
					     unsigned int selector);

int regulator_desc_list_voltage_linear(const struct regulator_desc *desc,
				       unsigned int selector);

#ifdef CONFIG_REGULATOR
const char *rdev_get_name(struct regulator_dev *rdev);
#else
static inline const char *rdev_get_name(struct regulator_dev *rdev)
{
	return NULL;
}
#endif

#endif
