/* SPDX-License-Identifier: GPL-2.0-or-later */

#ifndef LINUX_SERIAL_CORE_H
#define LINUX_SERIAL_CORE_H

#include <linux/bitops.h>
#include <linux/compiler.h>
#include <linux/console.h>
#include <linux/interrupt.h>
#include <linux/lockdep.h>
#include <linux/printk.h>
#include <linux/spinlock.h>
#include <linux/sched.h>
#include <linux/tty.h>
#include <linux/mutex.h>
#include <linux/sysrq.h>
#include <uapi/linux/serial_core.h>

#ifdef CONFIG_SERIAL_CORE_CONSOLE
#define uart_console(port) \
	((port)->cons && (port)->cons->index == (port)->line)
#else
#define uart_console(port)      ({ (void)port; 0; })
#endif

struct uart_port;
struct serial_struct;
struct serial_port_device;
struct device;
struct gpio_desc;


struct uart_ops {
	unsigned int	(*tx_empty)(struct uart_port *);
	void		(*set_mctrl)(struct uart_port *, unsigned int mctrl);
	unsigned int	(*get_mctrl)(struct uart_port *);
	void		(*stop_tx)(struct uart_port *);
	void		(*start_tx)(struct uart_port *);
	void		(*throttle)(struct uart_port *);
	void		(*unthrottle)(struct uart_port *);
	void		(*send_xchar)(struct uart_port *, char ch);
	void		(*stop_rx)(struct uart_port *);
	void		(*start_rx)(struct uart_port *);
	void		(*enable_ms)(struct uart_port *);
	void		(*break_ctl)(struct uart_port *, int ctl);
	int		(*startup)(struct uart_port *);
	void		(*shutdown)(struct uart_port *);
	void		(*flush_buffer)(struct uart_port *);
	void		(*set_termios)(struct uart_port *, struct ktermios *new,
				       const struct ktermios *old);
	void		(*set_ldisc)(struct uart_port *, struct ktermios *);
	void		(*pm)(struct uart_port *, unsigned int state,
			      unsigned int oldstate);
	const char	*(*type)(struct uart_port *);
	void		(*release_port)(struct uart_port *);
	int		(*request_port)(struct uart_port *);
	void		(*config_port)(struct uart_port *, int);
	int		(*verify_port)(struct uart_port *, struct serial_struct *);
	int		(*ioctl)(struct uart_port *, unsigned int, unsigned long);
#ifdef CONFIG_CONSOLE_POLL
	int		(*poll_init)(struct uart_port *);
	void		(*poll_put_char)(struct uart_port *, unsigned char);
	int		(*poll_get_char)(struct uart_port *);
#endif
};

#define NO_POLL_CHAR		0x00ff0000
#define UART_CONFIG_TYPE	(1 << 0)
#define UART_CONFIG_IRQ		(1 << 1)

struct uart_icount {
	__u32	cts;
	__u32	dsr;
	__u32	rng;
	__u32	dcd;
	__u32	rx;
	__u32	tx;
	__u32	frame;
	__u32	overrun;
	__u32	parity;
	__u32	brk;
	__u32	buf_overrun;
};

typedef u64 __bitwise upf_t;
typedef unsigned int __bitwise upstat_t;

struct uart_port {
	spinlock_t		lock;			
	unsigned long		iobase;			
	unsigned char __iomem	*membase;		
	unsigned int		(*serial_in)(struct uart_port *, int);
	void			(*serial_out)(struct uart_port *, int, int);
	void			(*set_termios)(struct uart_port *,
				               struct ktermios *new,
				               const struct ktermios *old);
	void			(*set_ldisc)(struct uart_port *,
					     struct ktermios *);
	unsigned int		(*get_mctrl)(struct uart_port *);
	void			(*set_mctrl)(struct uart_port *, unsigned int);
	unsigned int		(*get_divisor)(struct uart_port *,
					       unsigned int baud,
					       unsigned int *frac);
	void			(*set_divisor)(struct uart_port *,
					       unsigned int baud,
					       unsigned int quot,
					       unsigned int quot_frac);
	int			(*startup)(struct uart_port *port);
	void			(*shutdown)(struct uart_port *port);
	void			(*throttle)(struct uart_port *port);
	void			(*unthrottle)(struct uart_port *port);
	int			(*handle_irq)(struct uart_port *);
	void			(*pm)(struct uart_port *, unsigned int state,
				      unsigned int old);
	void			(*handle_break)(struct uart_port *);
	int			(*rs485_config)(struct uart_port *,
						struct ktermios *termios,
						struct serial_rs485 *rs485);
	int			(*iso7816_config)(struct uart_port *,
						  struct serial_iso7816 *iso7816);
	unsigned int		ctrl_id;		
	unsigned int		port_id;		
	unsigned int		irq;			
	unsigned long		irqflags;		
	unsigned int		uartclk;		
	unsigned int		fifosize;		
	unsigned char		x_char;			
	unsigned char		regshift;		

	unsigned char		iotype;			

#define UPIO_UNKNOWN		((unsigned char)~0U)	
#define UPIO_PORT		(SERIAL_IO_PORT)	
#define UPIO_HUB6		(SERIAL_IO_HUB6)	
#define UPIO_MEM		(SERIAL_IO_MEM)		
#define UPIO_MEM32		(SERIAL_IO_MEM32)	
#define UPIO_AU			(SERIAL_IO_AU)		
#define UPIO_TSI		(SERIAL_IO_TSI)		
#define UPIO_MEM32BE		(SERIAL_IO_MEM32BE)	
#define UPIO_MEM16		(SERIAL_IO_MEM16)	

	unsigned char		quirks;			

	
#define UPQ_NO_TXEN_TEST	BIT(0)

	unsigned int		read_status_mask;	
	unsigned int		ignore_status_mask;	
	struct uart_state	*state;			
	struct uart_icount	icount;			

	struct console		*cons;			
	
	upf_t			flags;

	
#ifdef CONFIG_HAS_IOPORT
#define UPF_FOURPORT		((__force upf_t) ASYNC_FOURPORT        )
#else
#define UPF_FOURPORT		0
#endif
#define UPF_SAK			((__force upf_t) ASYNC_SAK             )
#define UPF_SPD_HI		((__force upf_t) ASYNC_SPD_HI          )
#define UPF_SPD_VHI		((__force upf_t) ASYNC_SPD_VHI         )
#define UPF_SPD_CUST		((__force upf_t) ASYNC_SPD_CUST    )
#define UPF_SPD_WARP		((__force upf_t) ASYNC_SPD_WARP    )
#define UPF_SPD_MASK		((__force upf_t) ASYNC_SPD_MASK    )
#define UPF_SKIP_TEST		((__force upf_t) ASYNC_SKIP_TEST       )
#define UPF_AUTO_IRQ		((__force upf_t) ASYNC_AUTO_IRQ        )
#define UPF_HARDPPS_CD		((__force upf_t) ASYNC_HARDPPS_CD      )
#define UPF_SPD_SHI		((__force upf_t) ASYNC_SPD_SHI         )
#define UPF_LOW_LATENCY		((__force upf_t) ASYNC_LOW_LATENCY     )
#define UPF_BUGGY_UART		((__force upf_t) ASYNC_BUGGY_UART      )
#define UPF_MAGIC_MULTIPLIER	((__force upf_t) ASYNC_MAGIC_MULTIPLIER  )

#define UPF_NO_THRE_TEST	((__force upf_t) BIT_ULL(19))

#define UPF_AUTO_CTS		((__force upf_t) BIT_ULL(20))
#define UPF_AUTO_RTS		((__force upf_t) BIT_ULL(21))
#define UPF_HARD_FLOW		((__force upf_t) (UPF_AUTO_CTS | UPF_AUTO_RTS))

#define UPF_SOFT_FLOW		((__force upf_t) BIT_ULL(22))
#define UPF_CONS_FLOW		((__force upf_t) BIT_ULL(23))
#define UPF_SHARE_IRQ		((__force upf_t) BIT_ULL(24))
#define UPF_EXAR_EFR		((__force upf_t) BIT_ULL(25))
#define UPF_BUG_THRE		((__force upf_t) BIT_ULL(26))

#define UPF_FIXED_TYPE		((__force upf_t) BIT_ULL(27))
#define UPF_BOOT_AUTOCONF	((__force upf_t) BIT_ULL(28))
#define UPF_FIXED_PORT		((__force upf_t) BIT_ULL(29))
#define UPF_DEAD		((__force upf_t) BIT_ULL(30))
#define UPF_IOREMAP		((__force upf_t) BIT_ULL(31))
#define UPF_FULL_PROBE		((__force upf_t) BIT_ULL(32))

#define __UPF_CHANGE_MASK	0x17fff
#define UPF_CHANGE_MASK		((__force upf_t) __UPF_CHANGE_MASK)
#define UPF_USR_MASK		((__force upf_t) (UPF_SPD_MASK|UPF_LOW_LATENCY))

#if __UPF_CHANGE_MASK > ASYNC_FLAGS
#error Change mask not equivalent to userspace-visible bit defines
#endif

	
	upstat_t		status;

#define UPSTAT_CTS_ENABLE	((__force upstat_t) (1 << 0))
#define UPSTAT_DCD_ENABLE	((__force upstat_t) (1 << 1))
#define UPSTAT_AUTORTS		((__force upstat_t) (1 << 2))
#define UPSTAT_AUTOCTS		((__force upstat_t) (1 << 3))
#define UPSTAT_AUTOXOFF		((__force upstat_t) (1 << 4))
#define UPSTAT_SYNC_FIFO	((__force upstat_t) (1 << 5))

	bool			hw_stopped;		
	unsigned int		mctrl;			
	unsigned int		frame_time;		
	unsigned int		type;			
	const struct uart_ops	*ops;
	unsigned int		custom_divisor;
	unsigned int		line;			
	unsigned int		minor;
	resource_size_t		mapbase;		
	resource_size_t		mapsize;
	struct device		*dev;			
	struct serial_port_device *port_dev;		

	unsigned long		sysrq;			
	u8			sysrq_ch;		
	unsigned char		has_sysrq;
	unsigned char		sysrq_seq;		

	unsigned char		hub6;			
	unsigned char		suspended;
	unsigned char		console_reinit;
	const char		*name;			
	struct attribute_group	*attr_group;		
	const struct attribute_group **tty_groups;	
	struct serial_rs485     rs485;
	struct serial_rs485	rs485_supported;	
	struct gpio_desc	*rs485_term_gpio;	
	struct gpio_desc	*rs485_rx_during_tx_gpio; 
	struct serial_iso7816   iso7816;
	void			*private_data;		
};


static inline void __uart_port_lock_irqsave(struct uart_port *up, unsigned long *flags)
{
	spin_lock_irqsave(&up->lock, *flags);
}


static inline void __uart_port_unlock_irqrestore(struct uart_port *up, unsigned long flags)
{
	spin_unlock_irqrestore(&up->lock, flags);
}


static inline void uart_port_set_cons(struct uart_port *up, struct console *con)
{
	unsigned long flags;

	__uart_port_lock_irqsave(up, &flags);
	up->cons = con;
	__uart_port_unlock_irqrestore(up, flags);
}


static inline bool __uart_port_using_nbcon(struct uart_port *up)
{
	lockdep_assert_held_once(&up->lock);

	if (likely(!uart_console(up)))
		return false;

	
	if (hlist_unhashed_lockless(&up->cons->node) ||
	    !(up->cons->flags & CON_NBCON) ||
	    !up->cons->write_atomic) {
		return false;
	}

	return true;
}


static inline bool __uart_port_nbcon_try_acquire(struct uart_port *up)
{
	if (!__uart_port_using_nbcon(up))
		return true;

	return nbcon_device_try_acquire(up->cons);
}


static inline void __uart_port_nbcon_acquire(struct uart_port *up)
{
	if (!__uart_port_using_nbcon(up))
		return;

	while (!nbcon_device_try_acquire(up->cons))
		cpu_relax();
}


static inline void __uart_port_nbcon_release(struct uart_port *up)
{
	if (!__uart_port_using_nbcon(up))
		return;

	nbcon_device_release(up->cons);
}


static inline void uart_port_lock(struct uart_port *up)
{
	spin_lock(&up->lock);
	__uart_port_nbcon_acquire(up);
}


static inline void uart_port_lock_irq(struct uart_port *up)
{
	spin_lock_irq(&up->lock);
	__uart_port_nbcon_acquire(up);
}


static inline void uart_port_lock_irqsave(struct uart_port *up, unsigned long *flags)
{
	spin_lock_irqsave(&up->lock, *flags);
	__uart_port_nbcon_acquire(up);
}


static inline bool uart_port_trylock(struct uart_port *up)
{
	if (!spin_trylock(&up->lock))
		return false;

	if (!__uart_port_nbcon_try_acquire(up)) {
		spin_unlock(&up->lock);
		return false;
	}

	return true;
}


static inline bool uart_port_trylock_irqsave(struct uart_port *up, unsigned long *flags)
{
	if (!spin_trylock_irqsave(&up->lock, *flags))
		return false;

	if (!__uart_port_nbcon_try_acquire(up)) {
		spin_unlock_irqrestore(&up->lock, *flags);
		return false;
	}

	return true;
}


static inline void uart_port_unlock(struct uart_port *up)
{
	__uart_port_nbcon_release(up);
	spin_unlock(&up->lock);
}


static inline void uart_port_unlock_irq(struct uart_port *up)
{
	__uart_port_nbcon_release(up);
	spin_unlock_irq(&up->lock);
}


static inline void uart_port_unlock_irqrestore(struct uart_port *up, unsigned long flags)
{
	__uart_port_nbcon_release(up);
	spin_unlock_irqrestore(&up->lock, flags);
}

static inline int serial_port_in(struct uart_port *up, int offset)
{
	return up->serial_in(up, offset);
}

static inline void serial_port_out(struct uart_port *up, int offset, int value)
{
	up->serial_out(up, offset, value);
}


enum uart_pm_state {
	UART_PM_STATE_ON = 0,
	UART_PM_STATE_OFF = 3, 
	UART_PM_STATE_UNDEFINED,
};


struct uart_state {
	struct tty_port		port;

	enum uart_pm_state	pm_state;

	atomic_t		refcount;
	wait_queue_head_t	remove_wait;
	struct uart_port	*uart_port;
};

#define UART_XMIT_SIZE	PAGE_SIZE



#define WAKEUP_CHARS		256


static inline void uart_xmit_advance(struct uart_port *up, unsigned int chars)
{
	struct tty_port *tport = &up->state->port;

	kfifo_skip_count(&tport->xmit_fifo, chars);
	up->icount.tx += chars;
}

static inline unsigned int uart_fifo_out(struct uart_port *up,
		unsigned char *buf, unsigned int chars)
{
	struct tty_port *tport = &up->state->port;

	chars = kfifo_out(&tport->xmit_fifo, buf, chars);
	up->icount.tx += chars;

	return chars;
}

static inline unsigned int uart_fifo_get(struct uart_port *up,
		unsigned char *ch)
{
	struct tty_port *tport = &up->state->port;
	unsigned int chars;

	chars = kfifo_get(&tport->xmit_fifo, ch);
	up->icount.tx += chars;

	return chars;
}

struct module;
struct tty_driver;

struct uart_driver {
	struct module		*owner;
	const char		*driver_name;
	const char		*dev_name;
	int			 major;
	int			 minor;
	int			 nr;
	struct console		*cons;

	
	struct uart_state	*state;
	struct tty_driver	*tty_driver;
};

void uart_write_wakeup(struct uart_port *port);


enum UART_TX_FLAGS {
	UART_TX_NOSTOP = BIT(0),
};

#define __uart_port_tx(uport, ch, flags, tx_ready, put_char, tx_done,	      \
		       for_test, for_post)				      \
({									      \
	struct uart_port *__port = (uport);				      \
	struct tty_port *__tport = &__port->state->port;		      \
	unsigned int pending;						      \
									      \
	for (; (for_test) && (tx_ready); (for_post), __port->icount.tx++) {   \
		if (__port->x_char) {					      \
			(ch) = __port->x_char;				      \
			(put_char);					      \
			__port->x_char = 0;				      \
			continue;					      \
		}							      \
									      \
		if (uart_tx_stopped(__port))				      \
			break;						      \
									      \
		if (!kfifo_get(&__tport->xmit_fifo, &(ch)))		      \
			break;						      \
									      \
		(put_char);						      \
	}								      \
									      \
	(tx_done);							      \
									      \
	pending = kfifo_len(&__tport->xmit_fifo);			      \
	if (pending < WAKEUP_CHARS) {					      \
		uart_write_wakeup(__port);				      \
									      \
		if (!((flags) & UART_TX_NOSTOP) && pending == 0)	      \
			__port->ops->stop_tx(__port);			      \
	}								      \
									      \
	pending;							      \
})


#define uart_port_tx_limited(port, ch, count, tx_ready, put_char, tx_done) ({ \
	unsigned int __count = (count);					      \
	__uart_port_tx(port, ch, 0, tx_ready, put_char, tx_done, __count,     \
			__count--);					      \
})


#define uart_port_tx_limited_flags(port, ch, flags, count, tx_ready, put_char, tx_done) ({ \
	unsigned int __count = (count);							   \
	__uart_port_tx(port, ch, flags, tx_ready, put_char, tx_done, __count,		   \
			__count--);							   \
})


#define uart_port_tx(port, ch, tx_ready, put_char)			\
	__uart_port_tx(port, ch, 0, tx_ready, put_char, ({}), true, ({}))



#define uart_port_tx_flags(port, ch, flags, tx_ready, put_char)		\
	__uart_port_tx(port, ch, flags, tx_ready, put_char, ({}), true, ({}))

void uart_update_timeout(struct uart_port *port, unsigned int cflag,
			 unsigned int baud);
unsigned int uart_get_baud_rate(struct uart_port *port, struct ktermios *termios,
				const struct ktermios *old, unsigned int min,
				unsigned int max);
unsigned int uart_get_divisor(struct uart_port *port, unsigned int baud);


static inline unsigned long uart_fifo_timeout(struct uart_port *port)
{
	u64 fifo_timeout = (u64)READ_ONCE(port->frame_time) * port->fifosize;

	
	fifo_timeout += 20 * NSEC_PER_MSEC;

	return max(nsecs_to_jiffies(fifo_timeout), 1UL);
}


static inline unsigned long uart_poll_timeout(struct uart_port *port)
{
	unsigned long timeout = uart_fifo_timeout(port);

	return timeout > 6 ? (timeout / 2 - 2) : 1;
}


struct earlycon_device {
	struct console *con;
	struct uart_port port;
	char options[32];		
	unsigned int baud;
};

struct earlycon_id {
	char	name[15];
	char	name_term;	
	char	compatible[128];
	int	(*setup)(struct earlycon_device *, const char *options);
};

extern const struct earlycon_id __earlycon_table[];
extern const struct earlycon_id __earlycon_table_end[];

#if defined(CONFIG_SERIAL_EARLYCON) && !defined(MODULE)
#define EARLYCON_USED_OR_UNUSED	__used
#else
#define EARLYCON_USED_OR_UNUSED	__maybe_unused
#endif

#define OF_EARLYCON_DECLARE(_name, compat, fn)				\
	static const struct earlycon_id __UNIQUE_ID(__earlycon_##_name) \
		EARLYCON_USED_OR_UNUSED  __section("__earlycon_table")  \
		__aligned(__alignof__(struct earlycon_id))		\
		= { .name = __stringify(_name),				\
		    .compatible = compat,				\
		    .setup = fn }

#define EARLYCON_DECLARE(_name, fn)	OF_EARLYCON_DECLARE(_name, "", fn)

int of_setup_earlycon(const struct earlycon_id *match, unsigned long node,
		      const char *options);

#ifdef CONFIG_SERIAL_EARLYCON
extern bool earlycon_acpi_spcr_enable __initdata;
int setup_earlycon(char *buf);
#else
static const bool earlycon_acpi_spcr_enable EARLYCON_USED_OR_UNUSED;
static inline int setup_earlycon(char *buf) { return 0; }
#endif


static inline bool uart_console_registered_locked(struct uart_port *port)
{
	return uart_console(port) && console_is_registered_locked(port->cons);
}

static inline bool uart_console_registered(struct uart_port *port)
{
	return uart_console(port) && console_is_registered(port->cons);
}

struct uart_port *uart_get_console(struct uart_port *ports, int nr,
				   struct console *c);
int uart_parse_earlycon(char *p, unsigned char *iotype, resource_size_t *addr,
			char **options);
void uart_parse_options(const char *options, int *baud, int *parity, int *bits,
			int *flow);
int uart_set_options(struct uart_port *port, struct console *co, int baud,
		     int parity, int bits, int flow);
struct tty_driver *uart_console_device(struct console *co, int *index);
void uart_console_write(struct uart_port *port, const char *s,
			unsigned int count,
			void (*putchar)(struct uart_port *, unsigned char));


int uart_register_driver(struct uart_driver *uart);
void uart_unregister_driver(struct uart_driver *uart);
int uart_add_one_port(struct uart_driver *reg, struct uart_port *port);
void uart_remove_one_port(struct uart_driver *reg, struct uart_port *port);
int uart_read_port_properties(struct uart_port *port);
int uart_read_and_validate_port_properties(struct uart_port *port);
bool uart_match_port(const struct uart_port *port1,
		const struct uart_port *port2);


int uart_suspend_port(struct uart_driver *reg, struct uart_port *port);
int uart_resume_port(struct uart_driver *reg, struct uart_port *port);

static inline int uart_tx_stopped(struct uart_port *port)
{
	struct tty_struct *tty = port->state->port.tty;
	if ((tty && tty->flow.stopped) || port->hw_stopped)
		return 1;
	return 0;
}

static inline bool uart_cts_enabled(struct uart_port *uport)
{
	return !!(uport->status & UPSTAT_CTS_ENABLE);
}

static inline bool uart_softcts_mode(struct uart_port *uport)
{
	upstat_t mask = UPSTAT_CTS_ENABLE | UPSTAT_AUTOCTS;

	return ((uport->status & mask) == UPSTAT_CTS_ENABLE);
}



void uart_handle_dcd_change(struct uart_port *uport, bool active);
void uart_handle_cts_change(struct uart_port *uport, bool active);

void uart_insert_char(struct uart_port *port, unsigned int status,
		      unsigned int overrun, u8 ch, u8 flag);

void uart_xchar_out(struct uart_port *uport, int offset);

#ifdef CONFIG_MAGIC_SYSRQ_SERIAL
#define SYSRQ_TIMEOUT	(HZ * 5)

bool uart_try_toggle_sysrq(struct uart_port *port, u8 ch);

static inline int uart_handle_sysrq_char(struct uart_port *port, u8 ch)
{
	if (!port->sysrq)
		return 0;

	if (ch && time_before(jiffies, port->sysrq)) {
		if (sysrq_mask()) {
			handle_sysrq(ch);
			port->sysrq = 0;
			return 1;
		}
		if (uart_try_toggle_sysrq(port, ch))
			return 1;
	}
	port->sysrq = 0;

	return 0;
}

static inline int uart_prepare_sysrq_char(struct uart_port *port, u8 ch)
{
	if (!port->sysrq)
		return 0;

	if (ch && time_before(jiffies, port->sysrq)) {
		if (sysrq_mask()) {
			port->sysrq_ch = ch;
			port->sysrq = 0;
			return 1;
		}
		if (uart_try_toggle_sysrq(port, ch))
			return 1;
	}
	port->sysrq = 0;

	return 0;
}

static inline void uart_unlock_and_check_sysrq(struct uart_port *port)
{
	u8 sysrq_ch;

	if (!port->has_sysrq) {
		uart_port_unlock(port);
		return;
	}

	sysrq_ch = port->sysrq_ch;
	port->sysrq_ch = 0;

	uart_port_unlock(port);

	if (sysrq_ch)
		handle_sysrq(sysrq_ch);
}

static inline void uart_unlock_and_check_sysrq_irqrestore(struct uart_port *port,
		unsigned long flags)
{
	u8 sysrq_ch;

	if (!port->has_sysrq) {
		uart_port_unlock_irqrestore(port, flags);
		return;
	}

	sysrq_ch = port->sysrq_ch;
	port->sysrq_ch = 0;

	uart_port_unlock_irqrestore(port, flags);

	if (sysrq_ch)
		handle_sysrq(sysrq_ch);
}
#else	
static inline int uart_handle_sysrq_char(struct uart_port *port, u8 ch)
{
	return 0;
}
static inline int uart_prepare_sysrq_char(struct uart_port *port, u8 ch)
{
	return 0;
}
static inline void uart_unlock_and_check_sysrq(struct uart_port *port)
{
	uart_port_unlock(port);
}
static inline void uart_unlock_and_check_sysrq_irqrestore(struct uart_port *port,
		unsigned long flags)
{
	uart_port_unlock_irqrestore(port, flags);
}
#endif	


static inline int uart_handle_break(struct uart_port *port)
{
	struct uart_state *state = port->state;

	if (port->handle_break)
		port->handle_break(port);

#ifdef CONFIG_MAGIC_SYSRQ_SERIAL
	if (port->has_sysrq && uart_console(port)) {
		if (!port->sysrq) {
			port->sysrq = jiffies + SYSRQ_TIMEOUT;
			return 1;
		}
		port->sysrq = 0;
	}
#endif
	if (port->flags & UPF_SAK)
		do_SAK(state->port.tty);
	return 0;
}


#define UART_ENABLE_MS(port,cflag)	((port)->flags & UPF_HARDPPS_CD || \
					 (cflag) & CRTSCTS || \
					 !((cflag) & CLOCAL))

int uart_get_rs485_mode(struct uart_port *port);
#endif 
