/* SPDX-License-Identifier: GPL-2.0-or-later */


#ifndef __LINUX_NET_DSA_H
#define __LINUX_NET_DSA_H

#include <linux/if.h>
#include <linux/if_ether.h>
#include <linux/list.h>
#include <linux/notifier.h>
#include <linux/timer.h>
#include <linux/workqueue.h>
#include <linux/of.h>
#include <linux/ethtool.h>
#include <linux/net_tstamp.h>
#include <linux/phy.h>
#include <linux/platform_data/dsa.h>
#include <linux/phylink.h>
#include <net/devlink.h>
#include <net/switchdev.h>

struct dsa_8021q_context;
struct tc_action;

#define DSA_TAG_PROTO_NONE_VALUE		0
#define DSA_TAG_PROTO_BRCM_VALUE		1
#define DSA_TAG_PROTO_BRCM_PREPEND_VALUE	2
#define DSA_TAG_PROTO_DSA_VALUE			3
#define DSA_TAG_PROTO_EDSA_VALUE		4
#define DSA_TAG_PROTO_GSWIP_VALUE		5
#define DSA_TAG_PROTO_KSZ9477_VALUE		6
#define DSA_TAG_PROTO_KSZ9893_VALUE		7
#define DSA_TAG_PROTO_LAN9303_VALUE		8
#define DSA_TAG_PROTO_MTK_VALUE			9
#define DSA_TAG_PROTO_QCA_VALUE			10
#define DSA_TAG_PROTO_TRAILER_VALUE		11
#define DSA_TAG_PROTO_8021Q_VALUE		12
#define DSA_TAG_PROTO_SJA1105_VALUE		13
#define DSA_TAG_PROTO_KSZ8795_VALUE		14
#define DSA_TAG_PROTO_OCELOT_VALUE		15
#define DSA_TAG_PROTO_AR9331_VALUE		16
#define DSA_TAG_PROTO_RTL4_A_VALUE		17
#define DSA_TAG_PROTO_HELLCREEK_VALUE		18
#define DSA_TAG_PROTO_XRS700X_VALUE		19
#define DSA_TAG_PROTO_OCELOT_8021Q_VALUE	20
#define DSA_TAG_PROTO_SEVILLE_VALUE		21
#define DSA_TAG_PROTO_BRCM_LEGACY_VALUE		22
#define DSA_TAG_PROTO_SJA1110_VALUE		23
#define DSA_TAG_PROTO_RTL8_4_VALUE		24
#define DSA_TAG_PROTO_RTL8_4T_VALUE		25
#define DSA_TAG_PROTO_RZN1_A5PSW_VALUE		26
#define DSA_TAG_PROTO_LAN937X_VALUE		27
#define DSA_TAG_PROTO_VSC73XX_8021Q_VALUE	28

enum dsa_tag_protocol {
	DSA_TAG_PROTO_NONE		= DSA_TAG_PROTO_NONE_VALUE,
	DSA_TAG_PROTO_BRCM		= DSA_TAG_PROTO_BRCM_VALUE,
	DSA_TAG_PROTO_BRCM_LEGACY	= DSA_TAG_PROTO_BRCM_LEGACY_VALUE,
	DSA_TAG_PROTO_BRCM_PREPEND	= DSA_TAG_PROTO_BRCM_PREPEND_VALUE,
	DSA_TAG_PROTO_DSA		= DSA_TAG_PROTO_DSA_VALUE,
	DSA_TAG_PROTO_EDSA		= DSA_TAG_PROTO_EDSA_VALUE,
	DSA_TAG_PROTO_GSWIP		= DSA_TAG_PROTO_GSWIP_VALUE,
	DSA_TAG_PROTO_KSZ9477		= DSA_TAG_PROTO_KSZ9477_VALUE,
	DSA_TAG_PROTO_KSZ9893		= DSA_TAG_PROTO_KSZ9893_VALUE,
	DSA_TAG_PROTO_LAN9303		= DSA_TAG_PROTO_LAN9303_VALUE,
	DSA_TAG_PROTO_MTK		= DSA_TAG_PROTO_MTK_VALUE,
	DSA_TAG_PROTO_QCA		= DSA_TAG_PROTO_QCA_VALUE,
	DSA_TAG_PROTO_TRAILER		= DSA_TAG_PROTO_TRAILER_VALUE,
	DSA_TAG_PROTO_8021Q		= DSA_TAG_PROTO_8021Q_VALUE,
	DSA_TAG_PROTO_SJA1105		= DSA_TAG_PROTO_SJA1105_VALUE,
	DSA_TAG_PROTO_KSZ8795		= DSA_TAG_PROTO_KSZ8795_VALUE,
	DSA_TAG_PROTO_OCELOT		= DSA_TAG_PROTO_OCELOT_VALUE,
	DSA_TAG_PROTO_AR9331		= DSA_TAG_PROTO_AR9331_VALUE,
	DSA_TAG_PROTO_RTL4_A		= DSA_TAG_PROTO_RTL4_A_VALUE,
	DSA_TAG_PROTO_HELLCREEK		= DSA_TAG_PROTO_HELLCREEK_VALUE,
	DSA_TAG_PROTO_XRS700X		= DSA_TAG_PROTO_XRS700X_VALUE,
	DSA_TAG_PROTO_OCELOT_8021Q	= DSA_TAG_PROTO_OCELOT_8021Q_VALUE,
	DSA_TAG_PROTO_SEVILLE		= DSA_TAG_PROTO_SEVILLE_VALUE,
	DSA_TAG_PROTO_SJA1110		= DSA_TAG_PROTO_SJA1110_VALUE,
	DSA_TAG_PROTO_RTL8_4		= DSA_TAG_PROTO_RTL8_4_VALUE,
	DSA_TAG_PROTO_RTL8_4T		= DSA_TAG_PROTO_RTL8_4T_VALUE,
	DSA_TAG_PROTO_RZN1_A5PSW	= DSA_TAG_PROTO_RZN1_A5PSW_VALUE,
	DSA_TAG_PROTO_LAN937X		= DSA_TAG_PROTO_LAN937X_VALUE,
	DSA_TAG_PROTO_VSC73XX_8021Q	= DSA_TAG_PROTO_VSC73XX_8021Q_VALUE,
};

struct dsa_switch;

struct dsa_device_ops {
	struct sk_buff *(*xmit)(struct sk_buff *skb, struct net_device *dev);
	struct sk_buff *(*rcv)(struct sk_buff *skb, struct net_device *dev);
	void (*flow_dissect)(const struct sk_buff *skb, __be16 *proto,
			     int *offset);
	int (*connect)(struct dsa_switch *ds);
	void (*disconnect)(struct dsa_switch *ds);
	unsigned int needed_headroom;
	unsigned int needed_tailroom;
	const char *name;
	enum dsa_tag_protocol proto;
	
	bool promisc_on_conduit;
};

struct dsa_lag {
	struct net_device *dev;
	unsigned int id;
	struct mutex fdb_lock;
	struct list_head fdbs;
	refcount_t refcount;
};

struct dsa_switch_tree {
	struct list_head	list;

	
	struct list_head ports;

	
	struct raw_notifier_head	nh;

	
	unsigned int index;

	
	struct kref refcount;

	
	struct dsa_lag **lags;

	
	const struct dsa_device_ops *tag_ops;

	
	enum dsa_tag_protocol default_proto;

	
	bool setup;

	
	struct dsa_platform_data	*pd;

	
	struct list_head rtable;

	
	unsigned int lags_len;

	
	unsigned int last_switch;
};


#define dsa_lags_foreach_id(_id, _dst)				\
	for ((_id) = 1; (_id) <= (_dst)->lags_len; (_id)++)	\
		if ((_dst)->lags[(_id) - 1])

#define dsa_lag_foreach_port(_dp, _dst, _lag)			\
	list_for_each_entry((_dp), &(_dst)->ports, list)	\
		if (dsa_port_offloads_lag((_dp), (_lag)))

#define dsa_hsr_foreach_port(_dp, _ds, _hsr)			\
	list_for_each_entry((_dp), &(_ds)->dst->ports, list)	\
		if ((_dp)->ds == (_ds) && (_dp)->hsr_dev == (_hsr))

static inline struct dsa_lag *dsa_lag_by_id(struct dsa_switch_tree *dst,
					    unsigned int id)
{
	
	return dst->lags[id - 1];
}

static inline int dsa_lag_id(struct dsa_switch_tree *dst,
			     struct net_device *lag_dev)
{
	unsigned int id;

	dsa_lags_foreach_id(id, dst) {
		struct dsa_lag *lag = dsa_lag_by_id(dst, id);

		if (lag->dev == lag_dev)
			return lag->id;
	}

	return -ENODEV;
}


enum dsa_port_mall_action_type {
	DSA_PORT_MALL_MIRROR,
	DSA_PORT_MALL_POLICER,
};


struct dsa_mall_mirror_tc_entry {
	u8 to_local_port;
	bool ingress;
};


struct dsa_mall_policer_tc_entry {
	u32 burst;
	u64 rate_bytes_per_sec;
};


struct dsa_mall_tc_entry {
	struct list_head list;
	unsigned long cookie;
	enum dsa_port_mall_action_type type;
	union {
		struct dsa_mall_mirror_tc_entry mirror;
		struct dsa_mall_policer_tc_entry policer;
	};
};

struct dsa_bridge {
	struct net_device *dev;
	unsigned int num;
	bool tx_fwd_offload;
	refcount_t refcount;
};

struct dsa_port {
	
	union {
		struct net_device *conduit;
		struct net_device *user;
	};

	
	const struct dsa_device_ops *tag_ops;

	
	struct dsa_switch_tree *dst;
	struct sk_buff *(*rcv)(struct sk_buff *skb, struct net_device *dev);

	struct dsa_switch	*ds;

	unsigned int		index;

	enum {
		DSA_PORT_TYPE_UNUSED = 0,
		DSA_PORT_TYPE_CPU,
		DSA_PORT_TYPE_DSA,
		DSA_PORT_TYPE_USER,
	} type;

	const char		*name;
	struct dsa_port		*cpu_dp;
	u8			mac[ETH_ALEN];

	u8			stp_state;

	
	u8			vlan_filtering:1;

	
	u8			learning:1;

	u8			lag_tx_enabled:1;

	
	u8			conduit_admin_up:1;
	u8			conduit_oper_up:1;

	
	u8			cpu_port_in_lag:1;

	u8			setup:1;

	struct device_node	*dn;
	unsigned int		ageing_time;

	struct dsa_bridge	*bridge;
	struct devlink_port	devlink_port;
	struct phylink		*pl;
	struct phylink_config	pl_config;
	struct dsa_lag		*lag;
	struct net_device	*hsr_dev;

	struct list_head list;

	
	const struct ethtool_ops *orig_ethtool_ops;

	
	struct mutex		addr_lists_lock;
	struct list_head	fdbs;
	struct list_head	mdbs;

	struct mutex		vlans_lock;
	union {
		
		struct list_head	vlans;
		
		struct list_head	user_vlans;
	};
};

static inline struct dsa_port *
dsa_phylink_to_port(struct phylink_config *config)
{
	return container_of(config, struct dsa_port, pl_config);
}


struct dsa_link {
	struct dsa_port *dp;
	struct dsa_port *link_dp;
	struct list_head list;
};

enum dsa_db_type {
	DSA_DB_PORT,
	DSA_DB_LAG,
	DSA_DB_BRIDGE,
};

struct dsa_db {
	enum dsa_db_type type;

	union {
		const struct dsa_port *dp;
		struct dsa_lag lag;
		struct dsa_bridge bridge;
	};
};

struct dsa_mac_addr {
	unsigned char addr[ETH_ALEN];
	u16 vid;
	refcount_t refcount;
	struct list_head list;
	struct dsa_db db;
};

struct dsa_vlan {
	u16 vid;
	refcount_t refcount;
	struct list_head list;
};

struct dsa_switch {
	struct device *dev;

	
	struct dsa_switch_tree	*dst;
	unsigned int		index;

	
	u32			setup:1;

	
	u32			vlan_filtering_is_global:1;

	
	u32			needs_standalone_vlan_filtering:1;

	
	u32			configure_vlan_while_not_filtering:1;

	
	u32			untag_bridge_pvid:1;
	
	u32			untag_vlan_aware_bridge_pvid:1;

	
	u32			assisted_learning_on_cpu_port:1;

	
	u32			vlan_filtering:1;

	
	u32			mtu_enforcement_ingress:1;

	
	u32			fdb_isolation:1;

	
	u32			dscp_prio_mapping_is_global:1;

	
	struct notifier_block	nb;

	
	void *priv;

	void *tagger_data;

	
	struct dsa_chip_data	*cd;

	
	const struct dsa_switch_ops	*ops;

	
	const struct phylink_mac_ops	*phylink_mac_ops;

	
	u32			phys_mii_mask;
	struct mii_bus		*user_mii_bus;

	
	unsigned int ageing_time_min;
	unsigned int ageing_time_max;

	
	struct dsa_8021q_context *tag_8021q_ctx;

	
	struct devlink		*devlink;

	
	unsigned int		num_tx_queues;

	
	unsigned int		num_lag_ids;

	
	unsigned int		max_num_bridges;

	unsigned int		num_ports;
};

static inline struct dsa_port *dsa_to_port(struct dsa_switch *ds, int p)
{
	struct dsa_switch_tree *dst = ds->dst;
	struct dsa_port *dp;

	list_for_each_entry(dp, &dst->ports, list)
		if (dp->ds == ds && dp->index == p)
			return dp;

	return NULL;
}

static inline bool dsa_port_is_dsa(struct dsa_port *port)
{
	return port->type == DSA_PORT_TYPE_DSA;
}

static inline bool dsa_port_is_cpu(struct dsa_port *port)
{
	return port->type == DSA_PORT_TYPE_CPU;
}

static inline bool dsa_port_is_user(struct dsa_port *dp)
{
	return dp->type == DSA_PORT_TYPE_USER;
}

static inline bool dsa_port_is_unused(struct dsa_port *dp)
{
	return dp->type == DSA_PORT_TYPE_UNUSED;
}

static inline bool dsa_port_conduit_is_operational(struct dsa_port *dp)
{
	return dsa_port_is_cpu(dp) && dp->conduit_admin_up &&
	       dp->conduit_oper_up;
}

static inline bool dsa_is_unused_port(struct dsa_switch *ds, int p)
{
	return dsa_to_port(ds, p)->type == DSA_PORT_TYPE_UNUSED;
}

static inline bool dsa_is_cpu_port(struct dsa_switch *ds, int p)
{
	return dsa_to_port(ds, p)->type == DSA_PORT_TYPE_CPU;
}

static inline bool dsa_is_dsa_port(struct dsa_switch *ds, int p)
{
	return dsa_to_port(ds, p)->type == DSA_PORT_TYPE_DSA;
}

static inline bool dsa_is_user_port(struct dsa_switch *ds, int p)
{
	return dsa_to_port(ds, p)->type == DSA_PORT_TYPE_USER;
}

#define dsa_tree_for_each_user_port(_dp, _dst) \
	list_for_each_entry((_dp), &(_dst)->ports, list) \
		if (dsa_port_is_user((_dp)))

#define dsa_tree_for_each_user_port_continue_reverse(_dp, _dst) \
	list_for_each_entry_continue_reverse((_dp), &(_dst)->ports, list) \
		if (dsa_port_is_user((_dp)))

#define dsa_tree_for_each_cpu_port(_dp, _dst) \
	list_for_each_entry((_dp), &(_dst)->ports, list) \
		if (dsa_port_is_cpu((_dp)))

#define dsa_switch_for_each_port(_dp, _ds) \
	list_for_each_entry((_dp), &(_ds)->dst->ports, list) \
		if ((_dp)->ds == (_ds))

#define dsa_switch_for_each_port_safe(_dp, _next, _ds) \
	list_for_each_entry_safe((_dp), (_next), &(_ds)->dst->ports, list) \
		if ((_dp)->ds == (_ds))

#define dsa_switch_for_each_port_continue_reverse(_dp, _ds) \
	list_for_each_entry_continue_reverse((_dp), &(_ds)->dst->ports, list) \
		if ((_dp)->ds == (_ds))

#define dsa_switch_for_each_available_port(_dp, _ds) \
	dsa_switch_for_each_port((_dp), (_ds)) \
		if (!dsa_port_is_unused((_dp)))

#define dsa_switch_for_each_user_port(_dp, _ds) \
	dsa_switch_for_each_port((_dp), (_ds)) \
		if (dsa_port_is_user((_dp)))

#define dsa_switch_for_each_user_port_continue_reverse(_dp, _ds) \
	dsa_switch_for_each_port_continue_reverse((_dp), (_ds)) \
		if (dsa_port_is_user((_dp)))

#define dsa_switch_for_each_cpu_port(_dp, _ds) \
	dsa_switch_for_each_port((_dp), (_ds)) \
		if (dsa_port_is_cpu((_dp)))

#define dsa_switch_for_each_cpu_port_continue_reverse(_dp, _ds) \
	dsa_switch_for_each_port_continue_reverse((_dp), (_ds)) \
		if (dsa_port_is_cpu((_dp)))

static inline u32 dsa_user_ports(struct dsa_switch *ds)
{
	struct dsa_port *dp;
	u32 mask = 0;

	dsa_switch_for_each_user_port(dp, ds)
		mask |= BIT(dp->index);

	return mask;
}

static inline u32 dsa_cpu_ports(struct dsa_switch *ds)
{
	struct dsa_port *cpu_dp;
	u32 mask = 0;

	dsa_switch_for_each_cpu_port(cpu_dp, ds)
		mask |= BIT(cpu_dp->index);

	return mask;
}


static inline unsigned int dsa_routing_port(struct dsa_switch *ds, int device)
{
	struct dsa_switch_tree *dst = ds->dst;
	struct dsa_link *dl;

	list_for_each_entry(dl, &dst->rtable, list)
		if (dl->dp->ds == ds && dl->link_dp->ds->index == device)
			return dl->dp->index;

	return ds->num_ports;
}


static inline unsigned int dsa_towards_port(struct dsa_switch *ds, int device,
					    int port)
{
	if (device == ds->index)
		return port;
	else
		return dsa_routing_port(ds, device);
}


static inline unsigned int dsa_upstream_port(struct dsa_switch *ds, int port)
{
	const struct dsa_port *dp = dsa_to_port(ds, port);
	const struct dsa_port *cpu_dp = dp->cpu_dp;

	if (!cpu_dp)
		return port;

	return dsa_towards_port(ds, cpu_dp->ds->index, cpu_dp->index);
}


static inline bool dsa_is_upstream_port(struct dsa_switch *ds, int port)
{
	if (dsa_is_unused_port(ds, port))
		return false;

	return port == dsa_upstream_port(ds, port);
}


static inline bool dsa_is_downstream_port(struct dsa_switch *ds, int port)
{
	return dsa_is_dsa_port(ds, port) && !dsa_is_upstream_port(ds, port);
}


static inline unsigned int dsa_switch_upstream_port(struct dsa_switch *ds)
{
	struct dsa_port *dp;

	dsa_switch_for_each_available_port(dp, ds) {
		return dsa_upstream_port(ds, dp->index);
	}

	return ds->num_ports;
}


static inline bool dsa_switch_is_upstream_of(struct dsa_switch *upstream_ds,
					     struct dsa_switch *downstream_ds)
{
	int routing_port;

	if (upstream_ds == downstream_ds)
		return true;

	routing_port = dsa_routing_port(downstream_ds, upstream_ds->index);

	return dsa_is_upstream_port(downstream_ds, routing_port);
}

static inline bool dsa_port_is_vlan_filtering(const struct dsa_port *dp)
{
	const struct dsa_switch *ds = dp->ds;

	if (ds->vlan_filtering_is_global)
		return ds->vlan_filtering;
	else
		return dp->vlan_filtering;
}

static inline unsigned int dsa_port_lag_id_get(struct dsa_port *dp)
{
	return dp->lag ? dp->lag->id : 0;
}

static inline struct net_device *dsa_port_lag_dev_get(struct dsa_port *dp)
{
	return dp->lag ? dp->lag->dev : NULL;
}

static inline bool dsa_port_offloads_lag(struct dsa_port *dp,
					 const struct dsa_lag *lag)
{
	return dsa_port_lag_dev_get(dp) == lag->dev;
}

static inline struct net_device *dsa_port_to_conduit(const struct dsa_port *dp)
{
	if (dp->cpu_port_in_lag)
		return dsa_port_lag_dev_get(dp->cpu_dp);

	return dp->cpu_dp->conduit;
}

static inline
struct net_device *dsa_port_to_bridge_port(const struct dsa_port *dp)
{
	if (!dp->bridge)
		return NULL;

	if (dp->lag)
		return dp->lag->dev;
	else if (dp->hsr_dev)
		return dp->hsr_dev;

	return dp->user;
}

static inline struct net_device *
dsa_port_bridge_dev_get(const struct dsa_port *dp)
{
	return dp->bridge ? dp->bridge->dev : NULL;
}

static inline unsigned int dsa_port_bridge_num_get(struct dsa_port *dp)
{
	return dp->bridge ? dp->bridge->num : 0;
}

static inline bool dsa_port_bridge_same(const struct dsa_port *a,
					const struct dsa_port *b)
{
	struct net_device *br_a = dsa_port_bridge_dev_get(a);
	struct net_device *br_b = dsa_port_bridge_dev_get(b);

	
	return (!br_a || !br_b) ? false : (br_a == br_b);
}

static inline bool dsa_port_offloads_bridge_port(struct dsa_port *dp,
						 const struct net_device *dev)
{
	return dsa_port_to_bridge_port(dp) == dev;
}

static inline bool
dsa_port_offloads_bridge_dev(struct dsa_port *dp,
			     const struct net_device *bridge_dev)
{
	
	return dsa_port_bridge_dev_get(dp) == bridge_dev;
}

static inline bool dsa_port_offloads_bridge(struct dsa_port *dp,
					    const struct dsa_bridge *bridge)
{
	return dsa_port_bridge_dev_get(dp) == bridge->dev;
}


static inline bool dsa_tree_offloads_bridge_port(struct dsa_switch_tree *dst,
						 const struct net_device *dev)
{
	struct dsa_port *dp;

	list_for_each_entry(dp, &dst->ports, list)
		if (dsa_port_offloads_bridge_port(dp, dev))
			return true;

	return false;
}


static inline bool
dsa_tree_offloads_bridge_dev(struct dsa_switch_tree *dst,
			     const struct net_device *bridge_dev)
{
	struct dsa_port *dp;

	list_for_each_entry(dp, &dst->ports, list)
		if (dsa_port_offloads_bridge_dev(dp, bridge_dev))
			return true;

	return false;
}

static inline bool dsa_port_tree_same(const struct dsa_port *a,
				      const struct dsa_port *b)
{
	return a->ds->dst == b->ds->dst;
}

typedef int dsa_fdb_dump_cb_t(const unsigned char *addr, u16 vid,
			      bool is_static, void *data);
struct dsa_switch_ops {
	
	enum dsa_tag_protocol (*get_tag_protocol)(struct dsa_switch *ds,
						  int port,
						  enum dsa_tag_protocol mprot);
	int	(*change_tag_protocol)(struct dsa_switch *ds,
				       enum dsa_tag_protocol proto);
	
	int	(*connect_tag_protocol)(struct dsa_switch *ds,
					enum dsa_tag_protocol proto);

	int	(*port_change_conduit)(struct dsa_switch *ds, int port,
				       struct net_device *conduit,
				       struct netlink_ext_ack *extack);

	
	int	(*setup)(struct dsa_switch *ds);
	void	(*teardown)(struct dsa_switch *ds);

	
	int	(*port_setup)(struct dsa_switch *ds, int port);
	void	(*port_teardown)(struct dsa_switch *ds, int port);

	u32	(*get_phy_flags)(struct dsa_switch *ds, int port);

	
	int	(*phy_read)(struct dsa_switch *ds, int port, int regnum);
	int	(*phy_write)(struct dsa_switch *ds, int port,
			     int regnum, u16 val);

	
	void	(*phylink_get_caps)(struct dsa_switch *ds, int port,
				    struct phylink_config *config);
	void	(*phylink_fixed_state)(struct dsa_switch *ds, int port,
				       struct phylink_link_state *state);
	
	void	(*get_strings)(struct dsa_switch *ds, int port,
			       u32 stringset, uint8_t *data);
	void	(*get_ethtool_stats)(struct dsa_switch *ds,
				     int port, uint64_t *data);
	int	(*get_sset_count)(struct dsa_switch *ds, int port, int sset);
	void	(*get_ethtool_phy_stats)(struct dsa_switch *ds,
					 int port, uint64_t *data);
	void	(*get_eth_phy_stats)(struct dsa_switch *ds, int port,
				     struct ethtool_eth_phy_stats *phy_stats);
	void	(*get_eth_mac_stats)(struct dsa_switch *ds, int port,
				     struct ethtool_eth_mac_stats *mac_stats);
	void	(*get_eth_ctrl_stats)(struct dsa_switch *ds, int port,
				      struct ethtool_eth_ctrl_stats *ctrl_stats);
	void	(*get_rmon_stats)(struct dsa_switch *ds, int port,
				  struct ethtool_rmon_stats *rmon_stats,
				  const struct ethtool_rmon_hist_range **ranges);
	void	(*get_stats64)(struct dsa_switch *ds, int port,
				   struct rtnl_link_stats64 *s);
	void	(*get_pause_stats)(struct dsa_switch *ds, int port,
				   struct ethtool_pause_stats *pause_stats);
	void	(*self_test)(struct dsa_switch *ds, int port,
			     struct ethtool_test *etest, u64 *data);

	
	void	(*get_wol)(struct dsa_switch *ds, int port,
			   struct ethtool_wolinfo *w);
	int	(*set_wol)(struct dsa_switch *ds, int port,
			   struct ethtool_wolinfo *w);

	
	int	(*get_ts_info)(struct dsa_switch *ds, int port,
			       struct kernel_ethtool_ts_info *ts);

	
	int	(*get_mm)(struct dsa_switch *ds, int port,
			  struct ethtool_mm_state *state);
	int	(*set_mm)(struct dsa_switch *ds, int port,
			  struct ethtool_mm_cfg *cfg,
			  struct netlink_ext_ack *extack);
	void	(*get_mm_stats)(struct dsa_switch *ds, int port,
				struct ethtool_mm_stats *stats);

	
	int	(*port_get_default_prio)(struct dsa_switch *ds, int port);
	int	(*port_set_default_prio)(struct dsa_switch *ds, int port,
					 u8 prio);
	int	(*port_get_dscp_prio)(struct dsa_switch *ds, int port, u8 dscp);
	int	(*port_add_dscp_prio)(struct dsa_switch *ds, int port, u8 dscp,
				      u8 prio);
	int	(*port_del_dscp_prio)(struct dsa_switch *ds, int port, u8 dscp,
				      u8 prio);
	int	(*port_set_apptrust)(struct dsa_switch *ds, int port,
				     const u8 *sel, int nsel);
	int	(*port_get_apptrust)(struct dsa_switch *ds, int port, u8 *sel,
				     int *nsel);

	
	int	(*suspend)(struct dsa_switch *ds);
	int	(*resume)(struct dsa_switch *ds);

	
	int	(*port_enable)(struct dsa_switch *ds, int port,
			       struct phy_device *phy);
	void	(*port_disable)(struct dsa_switch *ds, int port);


	
	int	(*port_set_mac_address)(struct dsa_switch *ds, int port,
					const unsigned char *addr);

	
	struct dsa_port *(*preferred_default_local_cpu_port)(struct dsa_switch *ds);

	
	int	(*set_mac_eee)(struct dsa_switch *ds, int port,
			       struct ethtool_keee *e);
	int	(*get_mac_eee)(struct dsa_switch *ds, int port,
			       struct ethtool_keee *e);

	
	int	(*get_eeprom_len)(struct dsa_switch *ds);
	int	(*get_eeprom)(struct dsa_switch *ds,
			      struct ethtool_eeprom *eeprom, u8 *data);
	int	(*set_eeprom)(struct dsa_switch *ds,
			      struct ethtool_eeprom *eeprom, u8 *data);

	
	int	(*get_regs_len)(struct dsa_switch *ds, int port);
	void	(*get_regs)(struct dsa_switch *ds, int port,
			    struct ethtool_regs *regs, void *p);

	
	int	(*port_prechangeupper)(struct dsa_switch *ds, int port,
				       struct netdev_notifier_changeupper_info *info);

	
	int	(*set_ageing_time)(struct dsa_switch *ds, unsigned int msecs);
	int	(*port_bridge_join)(struct dsa_switch *ds, int port,
				    struct dsa_bridge bridge,
				    bool *tx_fwd_offload,
				    struct netlink_ext_ack *extack);
	void	(*port_bridge_leave)(struct dsa_switch *ds, int port,
				     struct dsa_bridge bridge);
	void	(*port_stp_state_set)(struct dsa_switch *ds, int port,
				      u8 state);
	int	(*port_mst_state_set)(struct dsa_switch *ds, int port,
				      const struct switchdev_mst_state *state);
	void	(*port_fast_age)(struct dsa_switch *ds, int port);
	int	(*port_vlan_fast_age)(struct dsa_switch *ds, int port, u16 vid);
	int	(*port_pre_bridge_flags)(struct dsa_switch *ds, int port,
					 struct switchdev_brport_flags flags,
					 struct netlink_ext_ack *extack);
	int	(*port_bridge_flags)(struct dsa_switch *ds, int port,
				     struct switchdev_brport_flags flags,
				     struct netlink_ext_ack *extack);
	void	(*port_set_host_flood)(struct dsa_switch *ds, int port,
				       bool uc, bool mc);

	
	int	(*port_vlan_filtering)(struct dsa_switch *ds, int port,
				       bool vlan_filtering,
				       struct netlink_ext_ack *extack);
	int	(*port_vlan_add)(struct dsa_switch *ds, int port,
				 const struct switchdev_obj_port_vlan *vlan,
				 struct netlink_ext_ack *extack);
	int	(*port_vlan_del)(struct dsa_switch *ds, int port,
				 const struct switchdev_obj_port_vlan *vlan);
	int	(*vlan_msti_set)(struct dsa_switch *ds, struct dsa_bridge bridge,
				 const struct switchdev_vlan_msti *msti);

	
	int	(*port_fdb_add)(struct dsa_switch *ds, int port,
				const unsigned char *addr, u16 vid,
				struct dsa_db db);
	int	(*port_fdb_del)(struct dsa_switch *ds, int port,
				const unsigned char *addr, u16 vid,
				struct dsa_db db);
	int	(*port_fdb_dump)(struct dsa_switch *ds, int port,
				 dsa_fdb_dump_cb_t *cb, void *data);
	int	(*lag_fdb_add)(struct dsa_switch *ds, struct dsa_lag lag,
			       const unsigned char *addr, u16 vid,
			       struct dsa_db db);
	int	(*lag_fdb_del)(struct dsa_switch *ds, struct dsa_lag lag,
			       const unsigned char *addr, u16 vid,
			       struct dsa_db db);

	
	int	(*port_mdb_add)(struct dsa_switch *ds, int port,
				const struct switchdev_obj_port_mdb *mdb,
				struct dsa_db db);
	int	(*port_mdb_del)(struct dsa_switch *ds, int port,
				const struct switchdev_obj_port_mdb *mdb,
				struct dsa_db db);
	
	int	(*get_rxnfc)(struct dsa_switch *ds, int port,
			     struct ethtool_rxnfc *nfc, u32 *rule_locs);
	int	(*set_rxnfc)(struct dsa_switch *ds, int port,
			     struct ethtool_rxnfc *nfc);

	
	int	(*cls_flower_add)(struct dsa_switch *ds, int port,
				  struct flow_cls_offload *cls, bool ingress);
	int	(*cls_flower_del)(struct dsa_switch *ds, int port,
				  struct flow_cls_offload *cls, bool ingress);
	int	(*cls_flower_stats)(struct dsa_switch *ds, int port,
				    struct flow_cls_offload *cls, bool ingress);
	int	(*port_mirror_add)(struct dsa_switch *ds, int port,
				   struct dsa_mall_mirror_tc_entry *mirror,
				   bool ingress, struct netlink_ext_ack *extack);
	void	(*port_mirror_del)(struct dsa_switch *ds, int port,
				   struct dsa_mall_mirror_tc_entry *mirror);
	int	(*port_policer_add)(struct dsa_switch *ds, int port,
				    struct dsa_mall_policer_tc_entry *policer);
	void	(*port_policer_del)(struct dsa_switch *ds, int port);
	int	(*port_setup_tc)(struct dsa_switch *ds, int port,
				 enum tc_setup_type type, void *type_data);

	
	int	(*crosschip_bridge_join)(struct dsa_switch *ds, int tree_index,
					 int sw_index, int port,
					 struct dsa_bridge bridge,
					 struct netlink_ext_ack *extack);
	void	(*crosschip_bridge_leave)(struct dsa_switch *ds, int tree_index,
					  int sw_index, int port,
					  struct dsa_bridge bridge);
	int	(*crosschip_lag_change)(struct dsa_switch *ds, int sw_index,
					int port);
	int	(*crosschip_lag_join)(struct dsa_switch *ds, int sw_index,
				      int port, struct dsa_lag lag,
				      struct netdev_lag_upper_info *info,
				      struct netlink_ext_ack *extack);
	int	(*crosschip_lag_leave)(struct dsa_switch *ds, int sw_index,
				       int port, struct dsa_lag lag);

	
	int	(*port_hwtstamp_get)(struct dsa_switch *ds, int port,
				     struct ifreq *ifr);
	int	(*port_hwtstamp_set)(struct dsa_switch *ds, int port,
				     struct ifreq *ifr);
	void	(*port_txtstamp)(struct dsa_switch *ds, int port,
				 struct sk_buff *skb);
	bool	(*port_rxtstamp)(struct dsa_switch *ds, int port,
				 struct sk_buff *skb, unsigned int type);

	
	int	(*devlink_param_get)(struct dsa_switch *ds, u32 id,
				     struct devlink_param_gset_ctx *ctx);
	int	(*devlink_param_set)(struct dsa_switch *ds, u32 id,
				     struct devlink_param_gset_ctx *ctx);
	int	(*devlink_info_get)(struct dsa_switch *ds,
				    struct devlink_info_req *req,
				    struct netlink_ext_ack *extack);
	int	(*devlink_sb_pool_get)(struct dsa_switch *ds,
				       unsigned int sb_index, u16 pool_index,
				       struct devlink_sb_pool_info *pool_info);
	int	(*devlink_sb_pool_set)(struct dsa_switch *ds, unsigned int sb_index,
				       u16 pool_index, u32 size,
				       enum devlink_sb_threshold_type threshold_type,
				       struct netlink_ext_ack *extack);
	int	(*devlink_sb_port_pool_get)(struct dsa_switch *ds, int port,
					    unsigned int sb_index, u16 pool_index,
					    u32 *p_threshold);
	int	(*devlink_sb_port_pool_set)(struct dsa_switch *ds, int port,
					    unsigned int sb_index, u16 pool_index,
					    u32 threshold,
					    struct netlink_ext_ack *extack);
	int	(*devlink_sb_tc_pool_bind_get)(struct dsa_switch *ds, int port,
					       unsigned int sb_index, u16 tc_index,
					       enum devlink_sb_pool_type pool_type,
					       u16 *p_pool_index, u32 *p_threshold);
	int	(*devlink_sb_tc_pool_bind_set)(struct dsa_switch *ds, int port,
					       unsigned int sb_index, u16 tc_index,
					       enum devlink_sb_pool_type pool_type,
					       u16 pool_index, u32 threshold,
					       struct netlink_ext_ack *extack);
	int	(*devlink_sb_occ_snapshot)(struct dsa_switch *ds,
					   unsigned int sb_index);
	int	(*devlink_sb_occ_max_clear)(struct dsa_switch *ds,
					    unsigned int sb_index);
	int	(*devlink_sb_occ_port_pool_get)(struct dsa_switch *ds, int port,
						unsigned int sb_index, u16 pool_index,
						u32 *p_cur, u32 *p_max);
	int	(*devlink_sb_occ_tc_port_bind_get)(struct dsa_switch *ds, int port,
						   unsigned int sb_index, u16 tc_index,
						   enum devlink_sb_pool_type pool_type,
						   u32 *p_cur, u32 *p_max);

	
	int	(*port_change_mtu)(struct dsa_switch *ds, int port,
				   int new_mtu);
	int	(*port_max_mtu)(struct dsa_switch *ds, int port);

	
	int	(*port_lag_change)(struct dsa_switch *ds, int port);
	int	(*port_lag_join)(struct dsa_switch *ds, int port,
				 struct dsa_lag lag,
				 struct netdev_lag_upper_info *info,
				 struct netlink_ext_ack *extack);
	int	(*port_lag_leave)(struct dsa_switch *ds, int port,
				  struct dsa_lag lag);

	
	int	(*port_hsr_join)(struct dsa_switch *ds, int port,
				 struct net_device *hsr,
				 struct netlink_ext_ack *extack);
	int	(*port_hsr_leave)(struct dsa_switch *ds, int port,
				  struct net_device *hsr);

	
	int	(*port_mrp_add)(struct dsa_switch *ds, int port,
				const struct switchdev_obj_mrp *mrp);
	int	(*port_mrp_del)(struct dsa_switch *ds, int port,
				const struct switchdev_obj_mrp *mrp);
	int	(*port_mrp_add_ring_role)(struct dsa_switch *ds, int port,
					  const struct switchdev_obj_ring_role_mrp *mrp);
	int	(*port_mrp_del_ring_role)(struct dsa_switch *ds, int port,
					  const struct switchdev_obj_ring_role_mrp *mrp);

	
	int	(*tag_8021q_vlan_add)(struct dsa_switch *ds, int port, u16 vid,
				      u16 flags);
	int	(*tag_8021q_vlan_del)(struct dsa_switch *ds, int port, u16 vid);

	
	void	(*conduit_state_change)(struct dsa_switch *ds,
					const struct net_device *conduit,
					bool operational);
};

#define DSA_DEVLINK_PARAM_DRIVER(_id, _name, _type, _cmodes)		\
	DEVLINK_PARAM_DRIVER(_id, _name, _type, _cmodes,		\
			     dsa_devlink_param_get, dsa_devlink_param_set, NULL)

int dsa_devlink_param_get(struct devlink *dl, u32 id,
			  struct devlink_param_gset_ctx *ctx);
int dsa_devlink_param_set(struct devlink *dl, u32 id,
			  struct devlink_param_gset_ctx *ctx,
			  struct netlink_ext_ack *extack);
int dsa_devlink_params_register(struct dsa_switch *ds,
				const struct devlink_param *params,
				size_t params_count);
void dsa_devlink_params_unregister(struct dsa_switch *ds,
				   const struct devlink_param *params,
				   size_t params_count);
int dsa_devlink_resource_register(struct dsa_switch *ds,
				  const char *resource_name,
				  u64 resource_size,
				  u64 resource_id,
				  u64 parent_resource_id,
				  const struct devlink_resource_size_params *size_params);

void dsa_devlink_resources_unregister(struct dsa_switch *ds);

void dsa_devlink_resource_occ_get_register(struct dsa_switch *ds,
					   u64 resource_id,
					   devlink_resource_occ_get_t *occ_get,
					   void *occ_get_priv);
void dsa_devlink_resource_occ_get_unregister(struct dsa_switch *ds,
					     u64 resource_id);
struct devlink_region *
dsa_devlink_region_create(struct dsa_switch *ds,
			  const struct devlink_region_ops *ops,
			  u32 region_max_snapshots, u64 region_size);
struct devlink_region *
dsa_devlink_port_region_create(struct dsa_switch *ds,
			       int port,
			       const struct devlink_port_region_ops *ops,
			       u32 region_max_snapshots, u64 region_size);
void dsa_devlink_region_destroy(struct devlink_region *region);

struct dsa_port *dsa_port_from_netdev(struct net_device *netdev);

struct dsa_devlink_priv {
	struct dsa_switch *ds;
};

static inline struct dsa_switch *dsa_devlink_to_ds(struct devlink *dl)
{
	struct dsa_devlink_priv *dl_priv = devlink_priv(dl);

	return dl_priv->ds;
}

static inline
struct dsa_switch *dsa_devlink_port_to_ds(struct devlink_port *port)
{
	struct devlink *dl = port->devlink;
	struct dsa_devlink_priv *dl_priv = devlink_priv(dl);

	return dl_priv->ds;
}

static inline int dsa_devlink_port_to_port(struct devlink_port *port)
{
	return port->index;
}

struct dsa_switch_driver {
	struct list_head	list;
	const struct dsa_switch_ops *ops;
};

bool dsa_fdb_present_in_other_db(struct dsa_switch *ds, int port,
				 const unsigned char *addr, u16 vid,
				 struct dsa_db db);
bool dsa_mdb_present_in_other_db(struct dsa_switch *ds, int port,
				 const struct switchdev_obj_port_mdb *mdb,
				 struct dsa_db db);


static inline bool netdev_uses_dsa(const struct net_device *dev)
{
#if IS_ENABLED(CONFIG_NET_DSA)
	return dev->dsa_ptr && dev->dsa_ptr->rcv;
#endif
	return false;
}


static inline void dsa_tag_generic_flow_dissect(const struct sk_buff *skb,
						__be16 *proto, int *offset)
{
#if IS_ENABLED(CONFIG_NET_DSA)
	const struct dsa_device_ops *ops = skb->dev->dsa_ptr->tag_ops;
	int tag_len = ops->needed_headroom;

	*offset = tag_len;
	*proto = ((__be16 *)skb->data)[(tag_len / 2) - 1];
#endif
}

void dsa_unregister_switch(struct dsa_switch *ds);
int dsa_register_switch(struct dsa_switch *ds);
void dsa_switch_shutdown(struct dsa_switch *ds);
struct dsa_switch *dsa_switch_find(int tree_index, int sw_index);
void dsa_flush_workqueue(void);
#ifdef CONFIG_PM_SLEEP
int dsa_switch_suspend(struct dsa_switch *ds);
int dsa_switch_resume(struct dsa_switch *ds);
#else
static inline int dsa_switch_suspend(struct dsa_switch *ds)
{
	return 0;
}
static inline int dsa_switch_resume(struct dsa_switch *ds)
{
	return 0;
}
#endif 

#if IS_ENABLED(CONFIG_NET_DSA)
bool dsa_user_dev_check(const struct net_device *dev);
#else
static inline bool dsa_user_dev_check(const struct net_device *dev)
{
	return false;
}
#endif

netdev_tx_t dsa_enqueue_skb(struct sk_buff *skb, struct net_device *dev);
void dsa_port_phylink_mac_change(struct dsa_switch *ds, int port, bool up);

#endif
