#ifndef LLC_H
#define LLC_H


#include <linux/if.h>
#include <linux/if_ether.h>
#include <linux/list.h>
#include <linux/spinlock.h>
#include <linux/rculist_nulls.h>
#include <linux/hash.h>
#include <linux/jhash.h>

#include <linux/atomic.h>

struct net_device;
struct packet_type;
struct sk_buff;

struct llc_addr {
	unsigned char lsap;
	unsigned char mac[IFHWADDRLEN];
};

#define LLC_SAP_STATE_INACTIVE	1
#define LLC_SAP_STATE_ACTIVE	2

#define LLC_SK_DEV_HASH_BITS 6
#define LLC_SK_DEV_HASH_ENTRIES (1<<LLC_SK_DEV_HASH_BITS)

#define LLC_SK_LADDR_HASH_BITS 6
#define LLC_SK_LADDR_HASH_ENTRIES (1<<LLC_SK_LADDR_HASH_BITS)


struct llc_sap {
	unsigned char	 state;
	unsigned char	 p_bit;
	unsigned char	 f_bit;
	refcount_t		 refcnt;
	int		 (*rcv_func)(struct sk_buff *skb,
				     struct net_device *dev,
				     struct packet_type *pt,
				     struct net_device *orig_dev);
	struct llc_addr	 laddr;
	struct list_head node;
	spinlock_t sk_lock;
	int sk_count;
	struct hlist_nulls_head sk_laddr_hash[LLC_SK_LADDR_HASH_ENTRIES];
	struct hlist_head sk_dev_hash[LLC_SK_DEV_HASH_ENTRIES];
	struct rcu_head rcu;
};

static inline
struct hlist_head *llc_sk_dev_hash(struct llc_sap *sap, int ifindex)
{
	u32 bucket = hash_32(ifindex, LLC_SK_DEV_HASH_BITS);

	return &sap->sk_dev_hash[bucket];
}

static inline
u32 llc_sk_laddr_hashfn(struct llc_sap *sap, const struct llc_addr *laddr)
{
	return hash_32(jhash(laddr->mac, sizeof(laddr->mac), 0),
		       LLC_SK_LADDR_HASH_BITS);
}

static inline
struct hlist_nulls_head *llc_sk_laddr_hash(struct llc_sap *sap,
					   const struct llc_addr *laddr)
{
	return &sap->sk_laddr_hash[llc_sk_laddr_hashfn(sap, laddr)];
}

#define LLC_DEST_INVALID         0      
#define LLC_DEST_SAP             1      
#define LLC_DEST_CONN            2      

extern struct list_head llc_sap_list;

int llc_rcv(struct sk_buff *skb, struct net_device *dev, struct packet_type *pt,
	    struct net_device *orig_dev);

int llc_mac_hdr_init(struct sk_buff *skb, const unsigned char *sa,
		     const unsigned char *da);

void llc_add_pack(int type,
		  void (*handler)(struct llc_sap *sap, struct sk_buff *skb));
void llc_remove_pack(int type);

void llc_set_station_handler(void (*handler)(struct sk_buff *skb));

struct llc_sap *llc_sap_open(unsigned char lsap,
			     int (*rcv)(struct sk_buff *skb,
					struct net_device *dev,
					struct packet_type *pt,
					struct net_device *orig_dev));
static inline void llc_sap_hold(struct llc_sap *sap)
{
	refcount_inc(&sap->refcnt);
}

static inline bool llc_sap_hold_safe(struct llc_sap *sap)
{
	return refcount_inc_not_zero(&sap->refcnt);
}

void llc_sap_close(struct llc_sap *sap);

static inline void llc_sap_put(struct llc_sap *sap)
{
	if (refcount_dec_and_test(&sap->refcnt))
		llc_sap_close(sap);
}

struct llc_sap *llc_sap_find(unsigned char sap_value);

int llc_build_and_send_ui_pkt(struct llc_sap *sap, struct sk_buff *skb,
			      const unsigned char *dmac, unsigned char dsap);

void llc_sap_handler(struct llc_sap *sap, struct sk_buff *skb);
void llc_conn_handler(struct llc_sap *sap, struct sk_buff *skb);

void llc_station_init(void);
void llc_station_exit(void);

#ifdef CONFIG_PROC_FS
int llc_proc_init(void);
void llc_proc_exit(void);
#else
#define llc_proc_init()	(0)
#define llc_proc_exit()	do { } while(0)
#endif 
#ifdef CONFIG_SYSCTL
int llc_sysctl_init(void);
void llc_sysctl_exit(void);

extern int sysctl_llc2_ack_timeout;
extern int sysctl_llc2_busy_timeout;
extern int sysctl_llc2_p_timeout;
extern int sysctl_llc2_rej_timeout;
#else
#define llc_sysctl_init() (0)
#define llc_sysctl_exit() do { } while(0)
#endif 
#endif 
