/* SPDX-License-Identifier: GPL-2.0 */


#ifndef _X25_H
#define _X25_H 
#include <linux/x25.h>
#include <linux/slab.h>
#include <linux/refcount.h>
#include <net/sock.h>

#define	X25_ADDR_LEN			16

#define	X25_MAX_L2_LEN			18	

#define	X25_STD_MIN_LEN			3
#define	X25_EXT_MIN_LEN			4

#define	X25_GFI_SEQ_MASK		0x30
#define	X25_GFI_STDSEQ			0x10
#define	X25_GFI_EXTSEQ			0x20

#define	X25_Q_BIT			0x80
#define	X25_D_BIT			0x40
#define	X25_STD_M_BIT			0x10
#define	X25_EXT_M_BIT			0x01

#define	X25_CALL_REQUEST		0x0B
#define	X25_CALL_ACCEPTED		0x0F
#define	X25_CLEAR_REQUEST		0x13
#define	X25_CLEAR_CONFIRMATION		0x17
#define	X25_DATA			0x00
#define	X25_INTERRUPT			0x23
#define	X25_INTERRUPT_CONFIRMATION	0x27
#define	X25_RR				0x01
#define	X25_RNR				0x05
#define	X25_REJ				0x09
#define	X25_RESET_REQUEST		0x1B
#define	X25_RESET_CONFIRMATION		0x1F
#define	X25_REGISTRATION_REQUEST	0xF3
#define	X25_REGISTRATION_CONFIRMATION	0xF7
#define	X25_RESTART_REQUEST		0xFB
#define	X25_RESTART_CONFIRMATION	0xFF
#define	X25_DIAGNOSTIC			0xF1
#define	X25_ILLEGAL			0xFD



#define	X25_COND_ACK_PENDING	0x01
#define	X25_COND_OWN_RX_BUSY	0x02
#define	X25_COND_PEER_RX_BUSY	0x04


enum {
	X25_STATE_0,		
	X25_STATE_1,		
	X25_STATE_2,		
	X25_STATE_3,		
	X25_STATE_4,		
	X25_STATE_5		
};

enum {
	X25_LINK_STATE_0,
	X25_LINK_STATE_1,
	X25_LINK_STATE_2,
	X25_LINK_STATE_3
};

#define X25_DEFAULT_T20		(180 * HZ)		
#define X25_DEFAULT_T21		(200 * HZ)		
#define X25_DEFAULT_T22		(180 * HZ)		
#define	X25_DEFAULT_T23		(180 * HZ)		
#define	X25_DEFAULT_T2		(3   * HZ)		

#define	X25_DEFAULT_WINDOW_SIZE	2			
#define	X25_DEFAULT_PACKET_SIZE	X25_PS128		
#define	X25_DEFAULT_THROUGHPUT	0x0A			
#define	X25_DEFAULT_REVERSE	0x00			

#define X25_SMODULUS 		8
#define	X25_EMODULUS		128



#define	X25_FAC_CLASS_MASK	0xC0

#define	X25_FAC_CLASS_A		0x00
#define	X25_FAC_CLASS_B		0x40
#define	X25_FAC_CLASS_C		0x80
#define	X25_FAC_CLASS_D		0xC0

#define	X25_FAC_REVERSE		0x01			
#define	X25_FAC_THROUGHPUT	0x02
#define	X25_FAC_PACKET_SIZE	0x42
#define	X25_FAC_WINDOW_SIZE	0x43

#define X25_MAX_FAC_LEN 	60
#define	X25_MAX_CUD_LEN		128

#define X25_FAC_CALLING_AE 	0xCB
#define X25_FAC_CALLED_AE 	0xC9

#define X25_MARKER 		0x00
#define X25_DTE_SERVICES 	0x0F
#define X25_MAX_AE_LEN 		40			
#define X25_MAX_DTE_FACIL_LEN	21			


#define X25_Q_BIT_FLAG		0
#define X25_INTERRUPT_FLAG	1
#define X25_ACCPT_APPRV_FLAG	2


struct x25_route {
	struct list_head	node;		
	struct x25_address	address;
	unsigned int		sigdigits;
	struct net_device	*dev;
	refcount_t		refcnt;
};

struct x25_neigh {
	struct list_head	node;
	struct net_device	*dev;
	unsigned int		state;
	unsigned int		extended;
	struct sk_buff_head	queue;
	unsigned long		t20;
	struct timer_list	t20timer;
	unsigned long		global_facil_mask;
	refcount_t		refcnt;
};

struct x25_sock {
	struct sock		sk;
	struct x25_address	source_addr, dest_addr;
	struct x25_neigh	*neighbour;
	unsigned int		lci, cudmatchlength;
	unsigned char		state, condition;
	unsigned short		vs, vr, va, vl;
	unsigned long		t2, t21, t22, t23;
	unsigned short		fraglen;
	unsigned long		flags;
	struct sk_buff_head	ack_queue;
	struct sk_buff_head	fragment_queue;
	struct sk_buff_head	interrupt_in_queue;
	struct sk_buff_head	interrupt_out_queue;
	struct timer_list	timer;
	struct x25_causediag	causediag;
	struct x25_facilities	facilities;
	struct x25_dte_facilities dte_facilities;
	struct x25_calluserdata	calluserdata;
	unsigned long 		vc_facil_mask;	
};

struct x25_forward {
	struct list_head	node;
	unsigned int		lci;
	struct net_device	*dev1;
	struct net_device	*dev2;
	atomic_t		refcnt;
};

#define x25_sk(ptr) container_of_const(ptr, struct x25_sock, sk)


extern int  sysctl_x25_restart_request_timeout;
extern int  sysctl_x25_call_request_timeout;
extern int  sysctl_x25_reset_request_timeout;
extern int  sysctl_x25_clear_request_timeout;
extern int  sysctl_x25_ack_holdback_timeout;
extern int  sysctl_x25_forward;

int x25_parse_address_block(struct sk_buff *skb,
			    struct x25_address *called_addr,
			    struct x25_address *calling_addr);

int x25_addr_ntoa(unsigned char *, struct x25_address *, struct x25_address *);
int x25_addr_aton(unsigned char *, struct x25_address *, struct x25_address *);
struct sock *x25_find_socket(unsigned int, struct x25_neigh *);
void x25_destroy_socket_from_timer(struct sock *);
int x25_rx_call_request(struct sk_buff *, struct x25_neigh *, unsigned int);
void x25_kill_by_neigh(struct x25_neigh *);


void x25_send_frame(struct sk_buff *, struct x25_neigh *);
int x25_lapb_receive_frame(struct sk_buff *, struct net_device *,
			   struct packet_type *, struct net_device *);
void x25_establish_link(struct x25_neigh *);
void x25_terminate_link(struct x25_neigh *);


int x25_parse_facilities(struct sk_buff *, struct x25_facilities *,
			 struct x25_dte_facilities *, unsigned long *);
int x25_create_facilities(unsigned char *, struct x25_facilities *,
			  struct x25_dte_facilities *, unsigned long);
int x25_negotiate_facilities(struct sk_buff *, struct sock *,
			     struct x25_facilities *,
			     struct x25_dte_facilities *);
void x25_limit_facilities(struct x25_facilities *, struct x25_neigh *);


void x25_clear_forward_by_lci(unsigned int lci);
void x25_clear_forward_by_dev(struct net_device *);
int x25_forward_data(int, struct x25_neigh *, struct sk_buff *);
int x25_forward_call(struct x25_address *, struct x25_neigh *, struct sk_buff *,
		     int);


int x25_process_rx_frame(struct sock *, struct sk_buff *);
int x25_backlog_rcv(struct sock *, struct sk_buff *);


void x25_link_control(struct sk_buff *, struct x25_neigh *, unsigned short);
void x25_link_device_up(struct net_device *);
void x25_link_device_down(struct net_device *);
void x25_link_established(struct x25_neigh *);
void x25_link_terminated(struct x25_neigh *);
void x25_transmit_clear_request(struct x25_neigh *, unsigned int,
				unsigned char);
void x25_transmit_link(struct sk_buff *, struct x25_neigh *);
int x25_subscr_ioctl(unsigned int, void __user *);
struct x25_neigh *x25_get_neigh(struct net_device *);
void x25_link_free(void);


static __inline__ void x25_neigh_hold(struct x25_neigh *nb)
{
	refcount_inc(&nb->refcnt);
}

static __inline__ void x25_neigh_put(struct x25_neigh *nb)
{
	if (refcount_dec_and_test(&nb->refcnt))
		kfree(nb);
}


int x25_output(struct sock *, struct sk_buff *);
void x25_kick(struct sock *);
void x25_enquiry_response(struct sock *);


struct x25_route *x25_get_route(struct x25_address *addr);
struct net_device *x25_dev_get(char *);
void x25_route_device_down(struct net_device *dev);
int x25_route_ioctl(unsigned int, void __user *);
void x25_route_free(void);

static __inline__ void x25_route_hold(struct x25_route *rt)
{
	refcount_inc(&rt->refcnt);
}

static __inline__ void x25_route_put(struct x25_route *rt)
{
	if (refcount_dec_and_test(&rt->refcnt))
		kfree(rt);
}


void x25_clear_queues(struct sock *);
void x25_frames_acked(struct sock *, unsigned short);
void x25_requeue_frames(struct sock *);
int x25_validate_nr(struct sock *, unsigned short);
void x25_write_internal(struct sock *, int);
int x25_decode(struct sock *, struct sk_buff *, int *, int *, int *, int *,
	       int *);
void x25_disconnect(struct sock *, int, unsigned char, unsigned char);


void x25_init_timers(struct sock *sk);
void x25_start_heartbeat(struct sock *);
void x25_start_t2timer(struct sock *);
void x25_start_t21timer(struct sock *);
void x25_start_t22timer(struct sock *);
void x25_start_t23timer(struct sock *);
void x25_stop_heartbeat(struct sock *);
void x25_stop_timer(struct sock *);
unsigned long x25_display_timer(struct sock *);
void x25_check_rbuf(struct sock *);


#ifdef CONFIG_SYSCTL
int x25_register_sysctl(void);
void x25_unregister_sysctl(void);
#else
static inline int x25_register_sysctl(void) { return 0; };
static inline void x25_unregister_sysctl(void) {};
#endif 

struct x25_skb_cb {
	unsigned int flags;
};
#define X25_SKB_CB(s) ((struct x25_skb_cb *) ((s)->cb))

extern struct hlist_head x25_list;
extern rwlock_t x25_list_lock;
extern struct list_head x25_route_list;
extern rwlock_t x25_route_list_lock;
extern struct list_head x25_forward_list;
extern rwlock_t x25_forward_list_lock;
extern struct list_head x25_neigh_list;
extern rwlock_t x25_neigh_list_lock;

int x25_proc_init(void);
void x25_proc_exit(void);
#endif
