/* SPDX-License-Identifier: GPL-2.0-or-later */
#ifndef __SOUND_CORE_H
#define __SOUND_CORE_H



#include <linux/device.h>
#include <linux/sched.h>		
#include <linux/mutex.h>		
#include <linux/rwsem.h>		
#include <linux/pm.h>			
#include <linux/stringify.h>
#include <linux/printk.h>
#include <linux/xarray.h>


#ifdef CONFIG_SND_DYNAMIC_MINORS
#define SNDRV_CARDS CONFIG_SND_MAX_CARDS
#else
#define SNDRV_CARDS 8		
#endif

#define CONFIG_SND_MAJOR	116	


struct pci_dev;
struct module;
struct completion;




enum snd_device_type {
	SNDRV_DEV_LOWLEVEL,
	SNDRV_DEV_INFO,
	SNDRV_DEV_BUS,
	SNDRV_DEV_CODEC,
	SNDRV_DEV_PCM,
	SNDRV_DEV_COMPRESS,
	SNDRV_DEV_RAWMIDI,
	SNDRV_DEV_TIMER,
	SNDRV_DEV_SEQUENCER,
	SNDRV_DEV_HWDEP,
	SNDRV_DEV_JACK,
	SNDRV_DEV_CONTROL,	
};

enum snd_device_state {
	SNDRV_DEV_BUILD,
	SNDRV_DEV_REGISTERED,
	SNDRV_DEV_DISCONNECTED,
};

struct snd_device;

struct snd_device_ops {
	int (*dev_free)(struct snd_device *dev);
	int (*dev_register)(struct snd_device *dev);
	int (*dev_disconnect)(struct snd_device *dev);
};

struct snd_device {
	struct list_head list;		
	struct snd_card *card;		
	enum snd_device_state state;	
	enum snd_device_type type;	
	void *device_data;		
	const struct snd_device_ops *ops;	
};

#define snd_device(n) list_entry(n, struct snd_device, list)



struct snd_card {
	int number;			

	char id[16];			
	char driver[16];		
	char shortname[32];		
	char longname[80];		
	char irq_descr[32];		
	char mixername[80];		
	char components[128];		
	struct module *module;		

	void *private_data;		
	void (*private_free) (struct snd_card *card); 
	struct list_head devices;	

	struct device *ctl_dev;		
	unsigned int last_numid;	
	struct rw_semaphore controls_rwsem;	
	rwlock_t controls_rwlock;	
	int controls_count;		
	size_t user_ctl_alloc_size;	// current memory allocation by user controls.
	struct list_head controls;	
	struct list_head ctl_files;	
#ifdef CONFIG_SND_CTL_FAST_LOOKUP
	struct xarray ctl_numids;	
	struct xarray ctl_hash;		
	bool ctl_hash_collision;	
#endif

	struct snd_info_entry *proc_root;	
	struct proc_dir_entry *proc_root_link;	

	struct list_head files_list;	
	struct snd_shutdown_f_ops *s_f_ops; 
	spinlock_t files_lock;		
	int shutdown;			
	struct completion *release_completion;
	struct device *dev;		
	struct device card_dev;		
	const struct attribute_group *dev_groups[4]; 
	bool registered;		
	bool managed;			
	bool releasing;			
	int sync_irq;			
	wait_queue_head_t remove_sleep;

	size_t total_pcm_alloc_bytes;	
	struct mutex memory_mutex;	
#ifdef CONFIG_SND_DEBUG
	struct dentry *debugfs_root;    
#endif

#ifdef CONFIG_PM
	unsigned int power_state;	
	atomic_t power_ref;
	wait_queue_head_t power_sleep;
	wait_queue_head_t power_ref_sleep;
#endif

#if IS_ENABLED(CONFIG_SND_MIXER_OSS)
	struct snd_mixer_oss *mixer_oss;
	int mixer_oss_change_count;
#endif
};

#define dev_to_snd_card(p)	container_of(p, struct snd_card, card_dev)

#ifdef CONFIG_PM
static inline unsigned int snd_power_get_state(struct snd_card *card)
{
	return READ_ONCE(card->power_state);
}

static inline void snd_power_change_state(struct snd_card *card, unsigned int state)
{
	WRITE_ONCE(card->power_state, state);
	wake_up(&card->power_sleep);
}


static inline void snd_power_ref(struct snd_card *card)
{
	atomic_inc(&card->power_ref);
}


static inline void snd_power_unref(struct snd_card *card)
{
	if (atomic_dec_and_test(&card->power_ref))
		wake_up(&card->power_ref_sleep);
}


static inline void snd_power_sync_ref(struct snd_card *card)
{
	wait_event(card->power_ref_sleep, !atomic_read(&card->power_ref));
}


int snd_power_wait(struct snd_card *card);
int snd_power_ref_and_wait(struct snd_card *card);

#else 

static inline int snd_power_wait(struct snd_card *card) { return 0; }
static inline void snd_power_ref(struct snd_card *card) {}
static inline void snd_power_unref(struct snd_card *card) {}
static inline int snd_power_ref_and_wait(struct snd_card *card) { return 0; }
static inline void snd_power_sync_ref(struct snd_card *card) {}
#define snd_power_get_state(card)	({ (void)(card); SNDRV_CTL_POWER_D0; })
#define snd_power_change_state(card, state)	do { (void)(card); } while (0)

#endif 

struct snd_minor {
	int type;			
	int card;			
	int device;			
	const struct file_operations *f_ops;	
	void *private_data;		
	struct device *dev;		
	struct snd_card *card_ptr;	
};


static inline struct device *snd_card_get_device_link(struct snd_card *card)
{
	return card ? &card->card_dev : NULL;
}



extern int snd_major;
extern int snd_ecards_limit;
extern const struct class sound_class;
#ifdef CONFIG_SND_DEBUG
extern struct dentry *sound_debugfs_root;
#endif

void snd_request_card(int card);

int snd_device_alloc(struct device **dev_p, struct snd_card *card);

int snd_register_device(int type, struct snd_card *card, int dev,
			const struct file_operations *f_ops,
			void *private_data, struct device *device);
int snd_unregister_device(struct device *dev);
void *snd_lookup_minor_data(unsigned int minor, int type);

#ifdef CONFIG_SND_OSSEMUL
int snd_register_oss_device(int type, struct snd_card *card, int dev,
			    const struct file_operations *f_ops, void *private_data);
int snd_unregister_oss_device(int type, struct snd_card *card, int dev);
void *snd_lookup_oss_minor_data(unsigned int minor, int type);
#endif

int snd_minor_info_init(void);



#ifdef CONFIG_SND_OSSEMUL
int snd_minor_info_oss_init(void);
#else
static inline int snd_minor_info_oss_init(void) { return 0; }
#endif



int copy_to_user_fromio(void __user *dst, const volatile void __iomem *src, size_t count);
int copy_from_user_toio(volatile void __iomem *dst, const void __user *src, size_t count);



int snd_card_locked(int card);
#if IS_ENABLED(CONFIG_SND_MIXER_OSS)
#define SND_MIXER_OSS_NOTIFY_REGISTER	0
#define SND_MIXER_OSS_NOTIFY_DISCONNECT	1
#define SND_MIXER_OSS_NOTIFY_FREE	2
extern int (*snd_mixer_oss_notify_callback)(struct snd_card *card, int cmd);
#endif

int snd_card_new(struct device *parent, int idx, const char *xid,
		 struct module *module, int extra_size,
		 struct snd_card **card_ret);
int snd_devm_card_new(struct device *parent, int idx, const char *xid,
		      struct module *module, size_t extra_size,
		      struct snd_card **card_ret);

void snd_card_disconnect(struct snd_card *card);
void snd_card_disconnect_sync(struct snd_card *card);
void snd_card_free(struct snd_card *card);
void snd_card_free_when_closed(struct snd_card *card);
int snd_card_free_on_error(struct device *dev, int ret);
void snd_card_set_id(struct snd_card *card, const char *id);
int snd_card_register(struct snd_card *card);
int snd_card_info_init(void);
int snd_card_add_dev_attr(struct snd_card *card,
			  const struct attribute_group *group);
int snd_component_add(struct snd_card *card, const char *component);
int snd_card_file_add(struct snd_card *card, struct file *file);
int snd_card_file_remove(struct snd_card *card, struct file *file);

struct snd_card *snd_card_ref(int card);


static inline void snd_card_unref(struct snd_card *card)
{
	put_device(&card->card_dev);
}

#define snd_card_set_dev(card, devptr) ((card)->dev = (devptr))



int snd_device_new(struct snd_card *card, enum snd_device_type type,
		   void *device_data, const struct snd_device_ops *ops);
int snd_device_register(struct snd_card *card, void *device_data);
int snd_device_register_all(struct snd_card *card);
void snd_device_disconnect(struct snd_card *card, void *device_data);
void snd_device_disconnect_all(struct snd_card *card);
void snd_device_free(struct snd_card *card, void *device_data);
void snd_device_free_all(struct snd_card *card);
int snd_device_get_state(struct snd_card *card, void *device_data);



#ifdef CONFIG_ISA_DMA_API
#define DMA_MODE_NO_ENABLE	0x0100

void snd_dma_program(unsigned long dma, unsigned long addr, unsigned int size, unsigned short mode);
void snd_dma_disable(unsigned long dma);
unsigned int snd_dma_pointer(unsigned long dma, unsigned int size);
int snd_devm_request_dma(struct device *dev, int dma, const char *name);
#endif


struct resource;
void release_and_free_resource(struct resource *res);



#ifdef CONFIG_SND_DEBUG

#define snd_BUG()		WARN(1, "BUG?\n")


#define snd_BUG_ON(cond)	WARN_ON((cond))

#else 

#define snd_BUG()			do { } while (0)

#define snd_BUG_ON(condition) ({ \
	int __ret_warn_on = !!(condition); \
	unlikely(__ret_warn_on); \
})

#endif 

#define SNDRV_OSS_VERSION         ((3<<16)|(8<<8)|(1<<4)|(0))	


#if IS_ENABLED(CONFIG_GAMEPORT)
#define gameport_set_dev_parent(gp,xdev) ((gp)->dev.parent = (xdev))
#define gameport_set_port_data(gp,r) ((gp)->port_data = (r))
#define gameport_get_port_data(gp) (gp)->port_data
#endif


struct snd_pci_quirk {
	unsigned short subvendor;	
	unsigned short subdevice;	
	unsigned short subdevice_mask;	
	int value;			
#ifdef CONFIG_SND_DEBUG_VERBOSE
	const char *name;		
#endif
};

#define _SND_PCI_QUIRK_ID_MASK(vend, mask, dev)	\
	.subvendor = (vend), .subdevice = (dev), .subdevice_mask = (mask)
#define _SND_PCI_QUIRK_ID(vend, dev) \
	_SND_PCI_QUIRK_ID_MASK(vend, 0xffff, dev)
#define SND_PCI_QUIRK_ID(vend,dev) {_SND_PCI_QUIRK_ID(vend, dev)}
#ifdef CONFIG_SND_DEBUG_VERBOSE
#define SND_PCI_QUIRK(vend,dev,xname,val) \
	{_SND_PCI_QUIRK_ID(vend, dev), .value = (val), .name = (xname)}
#define SND_PCI_QUIRK_VENDOR(vend, xname, val)			\
	{_SND_PCI_QUIRK_ID_MASK(vend, 0, 0), .value = (val), .name = (xname)}
#define SND_PCI_QUIRK_MASK(vend, mask, dev, xname, val)			\
	{_SND_PCI_QUIRK_ID_MASK(vend, mask, dev),			\
			.value = (val), .name = (xname)}
#define snd_pci_quirk_name(q)	((q)->name)
#else
#define SND_PCI_QUIRK(vend,dev,xname,val) \
	{_SND_PCI_QUIRK_ID(vend, dev), .value = (val)}
#define SND_PCI_QUIRK_MASK(vend, mask, dev, xname, val)			\
	{_SND_PCI_QUIRK_ID_MASK(vend, mask, dev), .value = (val)}
#define SND_PCI_QUIRK_VENDOR(vend, xname, val)			\
	{_SND_PCI_QUIRK_ID_MASK(vend, 0, 0), .value = (val)}
#define snd_pci_quirk_name(q)	""
#endif

#ifdef CONFIG_PCI
const struct snd_pci_quirk *
snd_pci_quirk_lookup(struct pci_dev *pci, const struct snd_pci_quirk *list);

const struct snd_pci_quirk *
snd_pci_quirk_lookup_id(u16 vendor, u16 device,
			const struct snd_pci_quirk *list);
#else
static inline const struct snd_pci_quirk *
snd_pci_quirk_lookup(struct pci_dev *pci, const struct snd_pci_quirk *list)
{
	return NULL;
}

static inline const struct snd_pci_quirk *
snd_pci_quirk_lookup_id(u16 vendor, u16 device,
			const struct snd_pci_quirk *list)
{
	return NULL;
}
#endif


struct snd_fasync;

int snd_fasync_helper(int fd, struct file *file, int on,
		      struct snd_fasync **fasyncp);
void snd_kill_fasync(struct snd_fasync *fasync, int signal, int poll);
void snd_fasync_free(struct snd_fasync *fasync);

#endif 
