/*
 *  linux/drivers/ide/fastata.c -- Amiga FastATA IDE Driver
 *
 *	Copyright (C) 2005 by Krystian Baclawski 
 *
 *  This driver is based on gayle.c by Geert Uytterhoeven
 *
 *  This file is subject to the terms and conditions of the GNU General Public
 *  License.  See the file COPYING in the main directory of this archive for
 *  more details.
 */

#include <linux/config.h>
#include <linux/types.h>
#include <linux/mm.h>
#include <linux/zorro.h>
#include <linux/interrupt.h>
#include <linux/blkdev.h>
#include <linux/hdreg.h>
#include <linux/ide.h>
#include <linux/init.h>

#include <asm/setup.h>
#include <asm/amigahw.h>
#include <asm/amigaints.h>
#include <asm/irq.h>

#include "ide_modes.h"
#include "ide-timing.h"

/* Base of the IDE interface */

#define FASTATA_PIO0_BASE	0xda2000
#define FASTATA_PIO3_BASE	0xdb2000
#define FASTATA_PIO4_BASE	0xdb6000
#define FASTATA_PIO5_BASE	0xdb4000

/* Offsets from one of the above bases */

#define FASTATA_PIO0_DATA		0x000
#define FASTATA_PIO0_ERROR		0x004		/* see err-bits */
#define FASTATA_PIO0_NSECTOR	0x008		/* nr of sectors to read/write */
#define FASTATA_PIO0_SECTOR		0x00C		/* starting sector */
#define FASTATA_PIO0_LCYL		0x010		/* starting cylinder */
#define FASTATA_PIO0_HCYL		0x014		/* high byte of starting cyl */
#define FASTATA_PIO0_SELECT		0x018		/* 101dhhhh , d=drive, hhhh=head */
#define FASTATA_PIO0_STATUS		0x01C		/* see status-bits */

#define FASTATA_DATALONG		0x000
#define FASTATA_ERROR			0x200		/* see err-bits */
#define FASTATA_DATA			0x208
#define FASTATA_NSECTOR			0x400		/* nr of sectors to read/write */
#define FASTATA_SECTOR			0x600		/* starting sector */
#define FASTATA_LCYL			0x800		/* starting cylinder */
#define FASTATA_HCYL			0xA00		/* high byte of starting cyl */
#define FASTATA_SELECT			0xC00		/* 101dhhhh , d=drive, hhhh=head */
#define FASTATA_STATUS			0xE00		/* see status-bits */

static int fastata_pio0_offsets[IDE_NR_PORTS] = {
    FASTATA_PIO0_DATA, FASTATA_PIO0_ERROR, FASTATA_PIO0_NSECTOR, FASTATA_PIO0_SECTOR, FASTATA_PIO0_LCYL,
    FASTATA_PIO0_HCYL, FASTATA_PIO0_SELECT, FASTATA_PIO0_STATUS, -1, -1
};

static int fastata_offsets[IDE_NR_PORTS] = {
    FASTATA_DATA, FASTATA_ERROR, FASTATA_NSECTOR, FASTATA_SECTOR, FASTATA_LCYL,
    FASTATA_HCYL, FASTATA_SELECT, FASTATA_STATUS, -1, -1
};

#define FASTATA_IDE_IRQ			0xda9000	/* interrupt */
#define FASTATA_IDEREG_SIZE		0x1000

/* Check and acknowledge the interrupt status */

static int fastata_ack_intr_a1200(ide_hwif_t *hwif)
{    
    if ((z_readb(hwif->io_ports[IDE_IRQ_OFFSET])) & 0x80)
	{
		// ATA-3: Status Register
		// Reading this register when an interrupt is pending
		// causes the interrupt to be cleared
        (void)z_readb(hwif->io_ports[IDE_STATUS_OFFSET]);

		z_writeb(0x7c, hwif->io_ports[IDE_IRQ_OFFSET]);
		
		return 1;
    }

    return 0;
}

/* Change register set  */

static void fastata_switch_register_set(ide_drive_t *drive, unsigned long phys_base, int *offsets)
{
    ide_ioreg_t base;
    ide_hwif_t  *hwif;
	unsigned int flags;
	int i;

    hwif = HWIF(drive);
    base = (ide_ioreg_t)ZTWO_VADDR(phys_base);

	if (hwif->hw.ack_intr && hwif->irq)
		disable_irq(hwif->irq);
	
	local_irq_save(flags);
	
	for (i = 0; i < IDE_NR_PORTS; i++)
	{
		if (offsets[i] == -1) {
			if (i == IDE_IRQ_OFFSET) {
				hwif->io_ports[i] = (ide_ioreg_t)ZTWO_VADDR(FASTATA_IDE_IRQ);
			} else {
				hwif->io_ports[i] = 0;
			}
		} else {
			hwif->io_ports[i] = base + offsets[i];
		}
	}

	for (i = 0; i < IDE_NR_PORTS; i++)
		hwif->hw.io_ports[i] = hwif->io_ports[i]; 

	local_irq_restore(flags);

	if (hwif->hw.ack_intr && hwif->irq)
	{
		hwif->hw.ack_intr(hwif);
    	enable_irq(hwif->irq);
	}

    //printk("FastATA: %s: register set switched to %08x\n", drive->name, (int)base);
}

/*  Switch to another drive  */

#if 0
static void fastata_select(ide_drive_t *drive)
{
    unsigned long phys_base;
    int *offsets;
	
	if (HWIF(drive)->hw.ack_intr && HWIF(drive)->irq)
		disable_irq(HWIF(drive)->irq);

    switch (drive->pio_speed) {
		case 0:
	    	phys_base = FASTATA_PIO0_BASE;
	    	offsets = fastata_pio0_offsets;
			break;
		case 1:
	    	phys_base = FASTATA_PIO3_BASE;
	    	offsets = fastata_offsets;
	    	break;
		case 2:
	    	phys_base = FASTATA_PIO4_BASE;
	    	offsets = fastata_offsets;
	    	break;
		case 3:
	    	phys_base = FASTATA_PIO5_BASE;
	    	offsets = fastata_offsets;
	    	break;
		default:
	    	panic("FastATA driver: drive->pio_speed trashed .\n");
	    	break;
    }

    fastata_switch_register_set(drive, phys_base, offsets);

	if (HWIF(drive)->hw.ack_intr && HWIF(drive)->irq)
	{
		HWIF(drive)->hw.ack_intr(HWIF(drive));
    	enable_irq(HWIF(drive)->irq);
	}
}
#endif

ide_drive_t* find_second_drive(ide_drive_t *drive)
{
	ide_drive_t *second;
	
	if (&(HWIF(drive)->drives[0]) == drive)
		second = &(HWIF(drive)->drives[1]);
	else
		second = &(HWIF(drive)->drives[0]);

	return (second->present) ? (second) : (0);
}

/* tune fastata pio modes */

static void fastata_tune(ide_drive_t *drive, byte mode_wanted)
{
    unsigned int	phys_base;
    unsigned int	*offsets;
	ide_drive_t 	*mate_drive;
	signed char		pio, speed;
	signed char		new_pio[] = {-1, -1};
	
	mate_drive = find_second_drive(drive);
	
	new_pio[0] = ide_get_best_pio_mode(drive, mode_wanted, 5, NULL);
	//printk("FastATA: drive (%s) can do pio mode %d.\n", drive->name, new_pio[0]);

	if (mate_drive)
	{
		new_pio[1] = ide_get_best_pio_mode(mate_drive, mode_wanted, 5, NULL);
		//printk("FastATA: mate drive (%s) can do pio mode %d.\n", mate_drive->name, new_pio[1]);
	}

	if ((new_pio[0] >= 0) && (new_pio[1] >= 0))
	{
		pio = min(new_pio[0], new_pio[1]);
	}
	else
	{
		if (new_pio[0] == -1)
			pio = new_pio[1];
		else
			pio = new_pio[0];
	}
	
	printk("FastATA: %s: drives will do pio mode %d.\n", HWIF(drive)->name, pio);

    switch (pio) {
		case 3:
	    	phys_base = FASTATA_PIO3_BASE;
	    	offsets = fastata_offsets;
	    	speed = XFER_PIO_3;
	    	drive->pio_speed = 1;
	    	break;
		case 4:
	    	phys_base = FASTATA_PIO4_BASE;
	    	offsets = fastata_offsets;
	    	speed = XFER_PIO_4;
	    	drive->pio_speed = 2;
	    	break;
		case 5:
	    	phys_base = FASTATA_PIO5_BASE;
	    	offsets = fastata_offsets;
	    	speed = XFER_PIO_5;
	    	drive->pio_speed = 3;
	    	break;
		default:
	    	phys_base = FASTATA_PIO0_BASE;
	    	offsets = fastata_pio0_offsets;
	    	speed = XFER_PIO_0;
	    	drive->pio_speed = 0;
	    	break;
    }

	{
		if (HWIF(drive)->hw.ack_intr && HWIF(drive)->irq)
			disable_irq(HWIF(drive)->irq);
		
		// Check if there is pending interrupt request and abandon it
	
		if (mate_drive)
		{
			printk("FastATA: %s: configuring speed.\n", mate_drive->name);
			
			mate_drive->io_32bit = 1;
			mate_drive->pio_speed = drive->pio_speed;
			
			ide_config_drive_speed(mate_drive, speed);
		}

		{
			printk("FastATA: %s: configuring speed.\n", drive->name);

			ide_config_drive_speed(drive, speed);
			drive->io_32bit = 1;
		}
		
		//printk("FastATA: %s: configuring register set.\n", HWIF(drive)->name);
		
    	fastata_switch_register_set(drive, phys_base, offsets);

		printk("FastATA: %s: configured to %08x.\n", HWIF(drive)->name, phys_base);
		
		if (HWIF(drive)->hw.ack_intr && HWIF(drive)->irq)
		{
			HWIF(drive)->hw.ack_intr(HWIF(drive));
			enable_irq(HWIF(drive)->irq);
		}
	}
}

/*  Probing for FastATA 1200 interface  */

static int fastata_probe(void)
{
    if (AMIGAHW_PRESENT(A1200_IDE))
    {
		ide_ioreg_t status1, status2;
        unsigned char v1, v2;

        request_mem_region(FASTATA_PIO0_BASE, FASTATA_IDEREG_SIZE, "IDE");
		request_mem_region(FASTATA_PIO3_BASE, FASTATA_IDEREG_SIZE, "IDE");

        status1 = (ide_ioreg_t)ZTWO_VADDR(FASTATA_PIO0_BASE + FASTATA_PIO0_STATUS);
        status2 = (ide_ioreg_t)ZTWO_VADDR(FASTATA_PIO3_BASE + FASTATA_STATUS);

        v1 = z_readb(status1) & 0xfd;
        v2 = z_readb(status2) & 0xfd;

        release_mem_region(FASTATA_PIO0_BASE, FASTATA_IDEREG_SIZE);
        release_mem_region(FASTATA_PIO3_BASE, FASTATA_IDEREG_SIZE);

        if (v1 == v2)
		{
    		printk("FastATA: A1200 IDE interface detected.\n");
    	    return 1;
		}
    }

    return 0;
}

/*  Initializing FastATA 1200 interface  */

void __init fastata_init(void)
{
    ide_ioreg_t base, irqport;
    hw_regs_t hw;
    int index;
	
	// Check if it's Amiga with
    if (!(MACH_IS_AMIGA && fastata_probe()))
		return;

    if (!(request_mem_region(FASTATA_PIO0_BASE, FASTATA_IDEREG_SIZE, "IDE") &&
		request_mem_region(FASTATA_PIO3_BASE, FASTATA_IDEREG_SIZE, "IDE") &&
		request_mem_region(FASTATA_PIO4_BASE, FASTATA_IDEREG_SIZE, "IDE") &&
		request_mem_region(FASTATA_PIO5_BASE, FASTATA_IDEREG_SIZE, "IDE")))
	return;

    base	= (ide_ioreg_t)ZTWO_VADDR(FASTATA_PIO0_BASE);
	irqport	= (ide_ioreg_t)ZTWO_VADDR(FASTATA_IDE_IRQ);
	
    ide_setup_ports(&hw, base, fastata_pio0_offsets, 0, irqport,
					fastata_ack_intr_a1200, IRQ_AMIGA_PORTS);

    index = ide_register_hw(&hw, NULL);

    if (index != -1)
    {
        printk("ide%d: FastATA 1200 IDE interface\n", index);

//		ide_hwifs[index].selectproc			= &fastata_select;
		ide_hwifs[index].tuneproc           = &fastata_tune;
		ide_hwifs[index].drives[0].autotune = 1;
        ide_hwifs[index].drives[1].autotune = 0;
		ide_hwifs[index].drives[0].pio_speed = 0;
        ide_hwifs[index].drives[1].pio_speed = 0;
    }
    else
    {
        release_mem_region(FASTATA_PIO0_BASE, FASTATA_IDEREG_SIZE);
        release_mem_region(FASTATA_PIO3_BASE, FASTATA_IDEREG_SIZE);
        release_mem_region(FASTATA_PIO4_BASE, FASTATA_IDEREG_SIZE);
        release_mem_region(FASTATA_PIO5_BASE, FASTATA_IDEREG_SIZE);
    }
}

/*
 * W ten sposb mona przecza argumenty na drugim urzdzeniu na tamie.
 *
 * HWIF(drive)->drives[!drive->select.b.unit].io_32bit = 1;
 *
 *
 */
